package key

import (
	"crypto/rand"
	"fmt"
	"time"
)

type EncryptionKey interface {
	NonceSize() int
	Encrypt(nonce []byte, plaintext []byte) (ciphertext []byte, err error)
	Decrypt(nonce []byte, ciphertext []byte) (plaintext []byte, err error)
	KeyType() KeyType
	Key() []byte
}

type SignatureKey interface {
	Sign(input []byte) ([]byte, error)
	Verify(input, sig []byte) (bool, error)
	KeyType() KeyType
	Key() []byte
	Public() (SignatureKey, error)
}

type GenerateOptions struct {
	KeyEpoch    uint32
	GeneratedAt time.Time

	EncryptionKeyType KeyType
	SignatureKeyType  KeyType
}

type KeySet struct {
	Keys []*Key `json:"keys,omitempty"`
}

type baseKey struct {
	// KeyEpoch
	KeyEpoch uint32 `json:"key_epoch"`

	// GeneratedAt
	GeneratedAt time.Time `json:"generated_at"`

	// Deactivated
	Deactivated bool `json:"deactivated,omitempty"`
}

type Key struct {
	baseKey
	EncryptionKey
	SignatureKey
}

func (k *baseKey) GetKeyEpoch() uint32 {
	return k.KeyEpoch
}

func (k *baseKey) GetGeneratedAt() time.Time {
	return k.GeneratedAt
}

func (k *baseKey) GetDeactivated() bool {
	return k.Deactivated
}

func (k *Key) ConsumerKey() (*Key, error) {
	var (
		err    error
		newKey = *k
	)
	newKey.SignatureKey, err = newKey.Public()
	if err != nil {
		return nil, err
	}
	return &newKey, err
}

func GenerateKey(options *GenerateOptions) (*Key, error) {
	var (
		err           error
		signatureKey  SignatureKey
		encryptionKey EncryptionKey
	)

	// we require encryption. This cannot be noop
	switch options.EncryptionKeyType {
	case KeyTypeSharedSecretBox:
		encryptionKey, err = GenerateSecretBoxKey()
		if err != nil {
			return nil, err
		}
	case KeyTypeSharedAESGCM:
		encryptionKey, err = GenerateAESGCMKey()
		if err != nil {
			return nil, err
		}
	default:
		return nil, fmt.Errorf("no support for generating encryption Key type (%s)", options.EncryptionKeyType)
	}

	switch options.SignatureKeyType {
	case KeyTypeNoop:
		signatureKey = noopKey{}
	case KeyTypePrivateEd25519:
		signatureKey, err = GenerateEd25519PrivateKey()
		if err != nil {
			return nil, err
		}
	default:
		return nil, fmt.Errorf("no support for generating signature Key type (%s)", options.EncryptionKeyType)
	}

	return &Key{
		baseKey: baseKey{
			KeyEpoch:    options.KeyEpoch,
			GeneratedAt: options.GeneratedAt,
		},
		EncryptionKey: encryptionKey,
		SignatureKey:  signatureKey,
	}, nil
}

func GenerateRandomNonce(nonceSize int) ([]byte, error) {
	nonce := make([]byte, nonceSize)
	_, err := rand.Read(nonce)
	if err != nil {
		return nil, err
	}
	return nonce, nil
}
