package key_test

import (
	"encoding/json"
	"fmt"
	"testing"
	"time"

	"code.justin.tv/amzn/StarfruitVault/key"

	"github.com/stretchr/testify/assert"
)

func TestShared(t *testing.T) {
	testTable := []struct {
		Name              string
		EncryptionKeyType key.KeyType
		SignatureKeyType  key.KeyType
	}{
		{
			EncryptionKeyType: key.KeyTypeSharedAESGCM,
			SignatureKeyType:  key.KeyTypeNoop,
		},
		{
			EncryptionKeyType: key.KeyTypeSharedSecretBox,
			SignatureKeyType:  key.KeyTypeNoop,
		},
		{
			EncryptionKeyType: key.KeyTypeSharedAESGCM,
			SignatureKeyType:  key.KeyTypePrivateEd25519,
		},
		{
			EncryptionKeyType: key.KeyTypeSharedSecretBox,
			SignatureKeyType:  key.KeyTypePrivateEd25519,
		},
	}

	for _, test := range testTable {
		t.Run(fmt.Sprintf("encryption(%s) signature(%s)", test.EncryptionKeyType, test.SignatureKeyType), func(t *testing.T) {
			k, err := key.GenerateKey(&key.GenerateOptions{
				KeyEpoch:          0,
				GeneratedAt:       time.Now(),
				EncryptionKeyType: test.EncryptionKeyType,
				SignatureKeyType:  test.SignatureKeyType,
			})
			assert.NoError(t, err)
			sharedEncryptionTests(t, k)
		})
	}
}

func sharedEncryptionTests(t *testing.T, k *key.Key) {
	msg := []byte("this is a super secure message")
	nonce, err := key.GenerateRandomNonce(k.NonceSize())
	assert.NoError(t, err)
	consumer, err := k.ConsumerKey()
	assert.NoError(t, err)

	t.Run("can encrypt and decrypt", func(t *testing.T) {
		assert := assert.New(t)
		encrypted, err := k.Encrypt(nonce, msg)
		assert.NoError(err)
		assert.NotEqual(msg, encrypted)

		decrypted, err := k.Decrypt(nonce, encrypted)
		assert.NoError(err)
		assert.Equal(msg, decrypted)
	})

	t.Run("can encrypt and decrypt after conversion", func(t *testing.T) {
		assert := assert.New(t)
		marshalled, err := json.Marshal(k)

		assert.NoError(err)

		var k key.Key
		err = json.Unmarshal(marshalled, &k)
		assert.NoError(err)

		encrypted, err := k.Encrypt(nonce, msg)
		assert.NoError(err)
		assert.NotEqual(msg, encrypted)

		decrypted, err := k.Decrypt(nonce, encrypted)
		assert.NoError(err)
		assert.Equal(msg, decrypted)
	})

	if k.SignatureKey.KeyType() == key.KeyTypeNoop {
		t.Run("sign is a noop", func(t *testing.T) {
			assert := assert.New(t)
			sig, err := k.Sign(msg)
			assert.NoError(err)
			assert.Empty(sig)
		})
		t.Run("verification succeeds as noop", func(t *testing.T) {
			assert := assert.New(t)
			ok, err := k.Verify(msg, msg)
			assert.NoError(err)
			assert.True(ok)
		})
	} else {
		t.Run("private Key can sign and verify", func(t *testing.T) {
			assert := assert.New(t)
			sig, err := k.Sign(msg)
			assert.NoError(err)

			ok, err := k.Verify(msg, sig)
			assert.NoError(err)
			assert.True(ok)
		})

		t.Run("consumer can verify", func(t *testing.T) {
			assert := assert.New(t)
			sig, err := k.Sign(msg)
			assert.NoError(err)

			ok, err := consumer.Verify(msg, sig)
			assert.NoError(err)
			assert.True(ok)
		})

		t.Run("consumer cannot sign", func(t *testing.T) {
			assert := assert.New(t)
			_, err = consumer.Sign(msg)
			assert.Error(err)
		})

		t.Run("verification fails on bad signature", func(t *testing.T) {
			assert := assert.New(t)
			sig, err := k.Sign(msg)
			assert.NoError(err)
			copy(sig[len(sig)/2:], sig[:len(sig)/2])

			ok, err := k.Verify(msg, sig)
			assert.NoError(err)
			assert.False(ok)
		})
	}
}
