package key

import (
	"encoding/json"
	"fmt"
)

type KeyType string

const (
	KeyTypeNoop            KeyType = ""
	KeyTypeSharedAESGCM    KeyType = "shared_aes_gcm"
	KeyTypePublicEd25519   KeyType = "public_ed25519"
	KeyTypePrivateEd25519  KeyType = "private_ed25519"
	KeyTypeSharedSecretBox KeyType = "shared_secretbox"
)

type serializedKey struct {
	baseKey
	// EncryptKeyType
	EncryptionKeyType KeyType `json:"encryption_key_type,omitempty"`

	// EncryptionKeyType
	EncryptionKey []byte `json:"encryption_key,omitempty"`

	// SignatureKeyType
	SignatureKeyType KeyType `json:"signature_key_type,omitempty"`

	// SignatureKey
	SignatureKey []byte `json:"signature_key,omitempty"`
}

var (
	_ json.Unmarshaler = (*Key)(nil)
	_ json.Marshaler   = (*Key)(nil)
)

func (k *Key) UnmarshalJSON(b []byte) error {
	var (
		err error
		s   serializedKey
	)
	err = json.Unmarshal(b, &s)
	if err != nil {
		return err
	}

	k.baseKey = s.baseKey

	// we require encryption. this cannot be noop
	switch s.EncryptionKeyType {
	case KeyTypeSharedSecretBox:
		k.EncryptionKey, err = newSecretBoxKey(s.EncryptionKey)
		if err != nil {
			return err
		}
	case KeyTypeSharedAESGCM:
		k.EncryptionKey, err = newAESGCMKey(s.EncryptionKey)
		if err != nil {
			return err
		}
	default:
		return fmt.Errorf("no support for encryption Key type (%s)", s.EncryptionKeyType)
	}

	switch s.SignatureKeyType {
	case KeyTypeNoop:
		k.SignatureKey = noopKey{}
	case KeyTypePublicEd25519:
		k.SignatureKey, err = newEd25519PublicKey(s.SignatureKey)
		if err != nil {
			return err
		}
	case KeyTypePrivateEd25519:
		k.SignatureKey, err = newEd25519PrivateKey(s.SignatureKey)
		if err != nil {
			return err
		}
	default:
		return fmt.Errorf("no support for signature Key type (%s)", s.EncryptionKeyType)
	}
	return nil
}

func (k *Key) MarshalJSON() ([]byte, error) {
	s := serializedKey{
		baseKey:           k.baseKey,
		EncryptionKeyType: k.EncryptionKey.KeyType(),
		EncryptionKey:     k.EncryptionKey.Key(),
		SignatureKeyType:  k.SignatureKey.KeyType(),
		SignatureKey:      k.SignatureKey.Key(),
	}
	return json.Marshal(&s)
}
