package key

import (
	"crypto/rand"
	"fmt"

	"golang.org/x/crypto/nacl/secretbox"
)

const (
	secretBoxNonceOverhead = 24
	secretBoxOverhead      = secretBoxNonceOverhead + secretbox.Overhead
)

type secretBoxKey struct {
	key [32]byte
}

func (k *secretBoxKey) KeyType() KeyType {
	return KeyTypeSharedSecretBox
}

func (k *secretBoxKey) Key() []byte {
	return k.key[:]
}

var _ EncryptionKey = (*secretBoxKey)(nil)

func GenerateSecretBoxKey() (*secretBoxKey, error) {
	var k secretBoxKey
	n, err := rand.Read(k.key[:])
	if err != nil {
		return nil, fmt.Errorf("unable to generate encryption Key: %w", err)
	}
	if n != len(k.key) {
		return nil, fmt.Errorf("unabled to read 32 bytes from crypto/rand")
	}

	return &k, nil
}

func newSecretBoxKey(key []byte) (*secretBoxKey, error) {
	newKey := &secretBoxKey{}
	copy(newKey.key[:], key)
	return newKey, nil
}

func (k *secretBoxKey) NonceSize() int {
	return secretBoxNonceOverhead
}

func (k *secretBoxKey) Encrypt(nonce []byte, plaintext []byte) ([]byte, error) {
	if len(nonce) != k.NonceSize() {
		return nil, fmt.Errorf("incorrect nonce size. expected (%d), got (%d)", k.NonceSize(), len(nonce))
	}
	var rawNonce [secretBoxNonceOverhead]byte
	copy(rawNonce[:], nonce)

	out := make([]byte, 0, secretBoxOverhead+len(plaintext))
	out = secretbox.Seal(out, plaintext, &rawNonce, &k.key)
	return out, nil
}

func (k *secretBoxKey) Decrypt(nonce []byte, input []byte) ([]byte, error) {
	if len(input) < secretBoxOverhead {
		return nil, fmt.Errorf("encrypted payload is too small")
	}

	var rawNonce [secretBoxNonceOverhead]byte
	copy(rawNonce[:], nonce)

	out, ok := secretbox.Open(nil, input, &rawNonce, &k.key)
	if !ok {
		return nil, fmt.Errorf("unable to decrypt payload")
	}
	return out, nil
}
