package aws

import (
	"context"
	"crypto/rand"
	"math/big"
	"sync"
	"time"

	"code.justin.tv/amzn/StarfruitVault/provider"

	"code.justin.tv/video/clock"

	"code.justin.tv/amzn/StarfruitVault/key"
)

type fetcher struct {
	client   *client
	hooks    provider.Hooks
	alias    string
	secretID string

	clock clock.Clock

	sync.Mutex
	lastSuccessfulFetch time.Time
}

func (f *fetcher) Fetch(ctx context.Context) (*key.KeySet, error) {
	var k key.KeySet
	err := f.client.Fetch(ctx, f.secretID, &k)
	if err != nil {
		return nil, err
	}

	f.Lock()
	defer f.Unlock()
	f.lastSuccessfulFetch = f.clock.Now()

	return &k, nil
}

func initialJitter() time.Duration {
	n, err := rand.Int(rand.Reader, big.NewInt(int64(pollInitialJitter)))
	if err != nil {
		return pollInitialJitter
	}
	return time.Duration(n.Int64())
}

func (f *fetcher) Run(ctx context.Context) error {
	timer := f.clock.Timer(initialJitter())
	defer timer.Stop()

	for {
		select {
		case <-ctx.Done():
			return ctx.Err()
		case <-timer.C:
			// The error here should have been reported already. We keep retrying to fetch a new one.
			_ = f.fetchAndReport(ctx)
			timer.Reset(pollInterval)
		}
	}
}

func (f *fetcher) RunAgeMonitor(ctx context.Context) error {
	ticker := f.clock.Ticker(monitorInterval)
	defer ticker.Stop()

	for {
		select {
		case <-ctx.Done():
			return ctx.Err()
		case <-ticker.C:
			// The error here should have been reported already. We keep retrying to fetch a new one.
			f.hooks.OnFetchAge(f.alias, f.clock.Since(f.GetLastSuccessfulTimestamp()))
		}
	}
}

func (f *fetcher) fetchAndReport(ctx context.Context) error {
	keys, err := f.Fetch(ctx)
	if err != nil {
		f.hooks.OnError(f.alias, err)
		return err
	}
	f.hooks.OnFetch(f.alias, keys)
	return nil
}

func (f *fetcher) GetLastSuccessfulTimestamp() time.Time {
	f.Lock()
	defer f.Unlock()
	return f.lastSuccessfulFetch
}

func (f *fetcher) GetSecretAlias() string {
	return f.alias
}

func (f *fetcher) GetSecretID() string {
	return f.secretID
}
