package aws

import (
	"context"
	"fmt"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/arn"

	"code.justin.tv/amzn/StarfruitVault/provider"

	"github.com/aws/aws-sdk-go/service/secretsmanager"
	"github.com/aws/aws-sdk-go/service/secretsmanager/secretsmanageriface"

	"github.com/aws/aws-sdk-go/aws/session"

	"code.justin.tv/video/clock"

	"code.justin.tv/video/invoker"
)

const (
	awsCurrent        = "AWSCURRENT"
	pollInterval      = 30 * time.Minute
	monitorInterval   = 5 * time.Minute
	pollInitialJitter = pollInterval
)

type Config struct {
	// Hooks provides callbacks for fetches,failures,age
	Hooks provider.Hooks

	// Session is the AWS Session. If this is nil, this will create a new session
	Session *session.Session

	// SecretIDs is a map from secretAlias -> aws SecretIDs
	SecretIDs map[string]string

	// used for mocking, not required to pass in usually
	SecretsManager secretsmanageriface.SecretsManagerAPI
	Clock          clock.Clock
}

type awsProvider struct {
	fetchers map[string]*fetcher
}

func NewAWSProvider(config *Config) (*awsProvider, error) {
	if config == nil {
		return nil, fmt.Errorf("config is nil")
	}

	if config.Session == nil {
		config.Session = session.Must(session.NewSession())
	}

	if config.Clock == nil {
		config.Clock = clock.New()
	}

	if config.Hooks == nil {
		return nil, fmt.Errorf("must provide hooks")
	}

	provider := &awsProvider{fetchers: make(map[string]*fetcher, len(config.SecretIDs))}
	for alias, secretID := range config.SecretIDs {
		client, err := getClient(config, secretID)
		if err != nil {
			return nil, err
		}
		provider.fetchers[alias] = &fetcher{
			client:   client,
			clock:    config.Clock,
			hooks:    config.Hooks,
			alias:    alias,
			secretID: secretID,
		}
	}

	return provider, nil
}

func getClient(config *Config, secretID string) (*client, error) {
	if config.SecretsManager != nil {
		return &client{config.SecretsManager}, nil
	}

	arn, err := arn.Parse(secretID)
	if err != nil {
		return nil, fmt.Errorf("invalid secretID: %w", err)
	}
	return &client{secretsmanager.New(config.Session, &aws.Config{Region: aws.String(arn.Region)})}, nil
}

func (p *awsProvider) Fetch(ctx context.Context) error {
	tasks := invoker.New()
	for _, f := range p.fetchers {
		tasks.Add(f.fetchAndReport)
	}
	return tasks.Run(ctx)
}

func (p *awsProvider) Run(ctx context.Context) error {
	tasks := invoker.New()
	for _, f := range p.fetchers {
		tasks.Add(f.Run, f.RunAgeMonitor)
	}
	return tasks.Run(ctx)
}
