package provider

import (
	"time"

	"code.justin.tv/amzn/StarfruitVault/key"
)

type (
	FetchHandler      func(secretAlias string, keys *key.KeySet)
	FetchAgeHandler   func(secretAlias string, age time.Duration)
	FetchErrorHandler func(secretAlias string, err error)
)

type Hooks interface {
	OnFetch(secretAlias string, keys *key.KeySet)
	OnFetchAge(secretAlias string, age time.Duration)
	OnError(secretAlias string, err error)

	RegisterOnFetch(FetchHandler)
	RegisterOnAge(FetchAgeHandler)
	RegisterOnError(FetchErrorHandler)
}

type hooks struct {
	fetchHandlers      []FetchHandler
	fetchAgeHandlers   []FetchAgeHandler
	fetchErrorHandlers []FetchErrorHandler
}

func (h *hooks) OnFetch(secretAlias string, keys *key.KeySet) {
	for _, handler := range h.fetchHandlers {
		handler(secretAlias, keys)
	}
}

func (h *hooks) OnFetchAge(secretAlias string, age time.Duration) {
	for _, handler := range h.fetchAgeHandlers {
		handler(secretAlias, age)
	}
}

func (h *hooks) OnError(secretAlias string, err error) {
	for _, handler := range h.fetchErrorHandlers {
		handler(secretAlias, err)
	}
}

func (h *hooks) RegisterOnFetch(handler FetchHandler) {
	h.fetchHandlers = append(h.fetchHandlers, handler)
}

func (h *hooks) RegisterOnAge(handler FetchAgeHandler) {
	h.fetchAgeHandlers = append(h.fetchAgeHandlers, handler)
}

func (h *hooks) RegisterOnError(handler FetchErrorHandler) {
	h.fetchErrorHandlers = append(h.fetchErrorHandlers, handler)
}

func NewHooks() *hooks {
	return &hooks{}
}
