package servicelog

import (
	"context"
	"fmt"
	"reflect"
	"strings"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"go.uber.org/zap"

	identifier "code.justin.tv/amzn/TwitchProcessIdentifier"
)

var defaultDimensionSets = [][]string{{"Region", "Service", "Stage"}, {"Region", "Service", "Stage", "Substage"}}

// Logger sends service logs to somewhere like cloudwatch or stdout
type Logger interface {
	Run(context.Context) error
	Send(MetricLogger) error
	NewLogEntry(string) *LogEntry
	SetStdoutLogger(*zap.Logger)
}

// MetricLogger collects metrics across dimensions
type MetricLogger interface {
	AddDimensionSet([]string)
	Metric(string, string) error
}

// LogEntry contains metrics defined in EMF format about a
// unit-of-work, such as an HTTP request or job read from a queue.
// It should be embedded into a user struct which includes fields intended
// to be logged to cloudwatch.
type LogEntry struct {
	Aws       awsRoot `json:"_aws"`
	Operation string  `json:"Operation,omitempty"`
	Region    string
	Service   string
	Stage     string
	Substage  string
}

type awsRoot struct {
	Timestamp         int64
	CloudWatchMetrics []metricDirective
}

type metricDefinition struct {
	Name string
	Unit string
}

type metricDirective struct {
	Namespace  string
	Dimensions [][]string
	Metrics    []metricDefinition
}

// Key types the value for inclusion in a Context
type key int

// ContextKey is the value for inclusion in a Context
const contextKey key = 0

// Context adds the user struct to a context
func Context(ctx context.Context, entry MetricLogger) context.Context {
	return context.WithValue(ctx, key(contextKey), entry)
}

// FromContext returns the user struct from a context
func FromContext(ctx context.Context) interface{} {
	return ctx.Value(key(contextKey))
}

// AddDimensionSet makes the metrics defined for the given log queryable by new dimensions
func (l *LogEntry) AddDimensionSet(set []string) {
	l.Aws.CloudWatchMetrics[0].Dimensions = append(l.Aws.CloudWatchMetrics[0].Dimensions, set)
}

// Metric registers a ServiceLog field as a metric. The given name must be a field declared at
// the root of the embedding struct.
func (l *LogEntry) Metric(name string, unit string) error {
	for _, metric := range l.Aws.CloudWatchMetrics[0].Metrics {
		if name == metric.Name {
			return fmt.Errorf("metric with name %v already defined", name)
		}
	}

	l.Aws.CloudWatchMetrics[0].Metrics = append(l.Aws.CloudWatchMetrics[0].Metrics, metricDefinition{
		Name: name,
		Unit: unit,
	})

	return nil
}

type logCreator struct {
	pid identifier.ProcessIdentifier
}

// newLogEntry creates a LogEntry from an operation with process identifying fields
func (l logCreator) NewLogEntry(opName string) *LogEntry {
	entry := &LogEntry{
		Aws: awsRoot{
			Timestamp: aws.TimeUnixMilli(time.Now()),
			CloudWatchMetrics: []metricDirective{
				{
					Namespace:  l.pid.Service,
					Dimensions: defaultDimensionSets,
				},
			},
		},
		Region:   l.pid.Region,
		Service:  l.pid.Service,
		Stage:    l.pid.Stage,
		Substage: l.pid.Substage,
	}
	if opName != "" {
		entry.Operation = opName
		entry.AddDimensionSet([]string{"Region", "Service", "Stage", "Substage", "Operation"})
	}
	return entry
}

// parseMetrics finds struct fields tagged with `servicelog` and appends
// the value into the EMF payload
func parseMetrics(v MetricLogger) error {
	if v == nil {
		return nil
	}

	sv := reflect.ValueOf(v)
	st := reflect.TypeOf(v)
	if st.Kind() == reflect.Ptr {
		sv = sv.Elem()
		st = st.Elem()
	}

	for i := 0; i < sv.NumField(); i++ {
		sf := st.Field(i)
		fv := sv.Field(i)
		tag := sf.Tag.Get("servicelog")
		if tag != "" {
			unit, opts := parseTag(tag)
			if opts.Contains("omitempty") && isEmptyValue(fv) {
				continue
			}

			err := v.Metric(sf.Name, unit)
			if err != nil {
				return err
			}
		}
	}

	return nil
}

func isEmptyValue(v reflect.Value) bool {
	switch v.Kind() {
	case reflect.Array, reflect.Map, reflect.Slice, reflect.String:
		return v.Len() == 0
	case reflect.Bool:
		return !v.Bool()
	case reflect.Int, reflect.Int8, reflect.Int16, reflect.Int32, reflect.Int64:
		return v.Int() == 0
	case reflect.Uint, reflect.Uint8, reflect.Uint16, reflect.Uint32, reflect.Uint64, reflect.Uintptr:
		return v.Uint() == 0
	case reflect.Float32, reflect.Float64:
		return v.Float() == 0
	case reflect.Interface, reflect.Ptr:
		return v.IsNil()
	}
	return false
}

type tagOptions string

func parseTag(tag string) (string, tagOptions) {
	if idx := strings.Index(tag, ","); idx != -1 {
		return tag[:idx], tagOptions(tag[idx+1:])
	}
	return tag, tagOptions("")
}

func (o tagOptions) Contains(optionName string) bool {
	if len(o) == 0 {
		return false
	}
	s := string(o)
	for s != "" {
		var next string
		i := strings.Index(s, ",")
		if i >= 0 {
			s, next = s[:i], s[i+1:]
		}
		if s == optionName {
			return true
		}
		s = next
	}
	return false
}
