package servicelog

import (
	"context"

	"github.com/twitchtv/twirp"
)

// CreateFunc is given a LogEntry and should return a struct which embeds it
type CreateFunc func(context.Context, *LogEntry) MetricLogger

// FinalizeFunc returns the embedding struct to send to cloudwatch
type FinalizeFunc func(context.Context) MetricLogger

// ErrorFunc allows the embedding struct to mark relevant error fields
type ErrorFunc func(context.Context, twirp.Error) context.Context

// ServerHooks enable Twirp services to automatically create and send service logs per request
func ServerHooks(l Logger, createFunc CreateFunc, finalizeFunc FinalizeFunc, errorFunc ErrorFunc) *twirp.ServerHooks {
	return &twirp.ServerHooks{
		RequestRouted: func(ctx context.Context) (context.Context, error) {
			op, _ := twirp.MethodName(ctx)

			entry := l.NewLogEntry(op)

			wrapped := createFunc(ctx, entry)
			ctx = Context(ctx, wrapped)
			return ctx, nil
		},
		ResponseSent: func(ctx context.Context) {
			wrapped := finalizeFunc(ctx)
			l.Send(wrapped)
		},
		Error: errorFunc,
	}
}
