package cache

import (
	telemetry "code.justin.tv/amzn/TwitchTelemetry"
	"fmt"
	"time"

	lru "github.com/hashicorp/golang-lru"
)

type Cache interface {
	// Get returns the key's value from the cache and update the recency of the item
	Get(key string) (interface{}, bool)
	// GetWithExpiration returns the key's value and the expiration time from the cache and update the recency of the item
	// the returned expiration time is in local timezone
	GetWithExpiration(key string) (interface{}, time.Time, bool)
	// Peek returns the key's value from the cache without updating the recency
	Peek(key string) (interface{}, bool)
	// PeekWithExpiration returns the key's value and its expiration from the cache without updating recency
	// the returned expiration time is in local timezone
	PeekWithExpiration(key string) (interface{}, time.Time, bool)
	// Set adds or updates an item in the cache, the item will not expire
	Set(key string, val interface{}) bool
	// SetDefault adds or updates an item in the cache by setting an expiration time
	SetWithExpiration(key string, val interface{}, d time.Duration) bool
	// Delete removes an item from the cache. It is a noop if the item does not exists
	Delete(key string)
	// Size return the number of items in the cache
	Size() int
}

var _ Cache = (*cacheImpl)(nil)

type cacheImpl struct {
	lruCache *lru.Cache
	config   Config
	reporter *telemetry.SampleReporter
}

type item struct {
	val        interface{}
	expiration int64
}

func NewCache(config Config) (*cacheImpl, error) {
	config = FillInDefault(config)
	var evictFunc func(interface{}, interface{})
	if config.OnEvicted != nil {
		evictFunc = func(key interface{}, val interface{}) {
			config.OnEvicted(key.(string), val)
		}
	}
	cache, err := lru.NewWithEvict(config.CacheSize, evictFunc)
	if err != nil {
		return nil, fmt.Errorf("error creating cache, %w", err)
	}
	return &cacheImpl{
		lruCache: cache,
		config:   config,
		reporter: config.MetricReporter,
	}, nil
}

func (c *cacheImpl) Get(key string) (interface{}, bool) {
	val, _, ok := c.GetWithExpiration(key)
	return val, ok
}

func (c *cacheImpl) GetWithExpiration(key string) (interface{}, time.Time, bool) {
	val, ok := c.lruCache.Get(key)
	if !ok {
		c.reportMiss()
		return nil, time.Time{}, false
	}

	if item, ok := val.(*item); ok {
		if item.expiration == 0 {
			c.reportHit()
			return item.val, time.Time{}, true
		}
		if item.expiration > 0 && c.config.CustomClock.GetCurrentTime() <= item.expiration {
			c.reportHit()
			return item.val, time.Unix(0, item.expiration), true
		}
	}

	c.reportMiss()
	return nil, time.Time{}, false
}

func (c *cacheImpl) PeekWithExpiration(key string) (interface{}, time.Time, bool) {
	val, ok := c.lruCache.Peek(key)
	if !ok {
		c.reportMiss()
		return nil, time.Time{}, false
	}

	if item, ok := val.(*item); ok {
		if item.expiration == 0 {
			c.reportHit()
			return item.val, time.Time{}, true
		}
		if item.expiration > 0 && c.config.CustomClock.GetCurrentTime() <= item.expiration {
			c.reportHit()
			return item.val, time.Unix(0, item.expiration), true
		}
	}

	c.reportMiss()
	return nil, time.Time{}, false
}

func (c *cacheImpl) Peek(key string) (interface{}, bool) {
	val, _, ok := c.PeekWithExpiration(key)
	return val, ok
}

func (c *cacheImpl) Set(key string, val interface{}) bool {
	return c.SetWithExpiration(key, val, 0)
}

func (c *cacheImpl) SetWithExpiration(key string, val interface{}, d time.Duration) bool {
	var e int64
	if d > 0 {
		e = c.config.CustomClock.GetCurrentTime() + d.Nanoseconds()
	}

	item := item{
		val:        val,
		expiration: e,
	}

	evicted := c.lruCache.Add(key, &item)
	return evicted
}

func (c *cacheImpl) Delete(key string) {
	c.lruCache.Remove(key)
}

func (c *cacheImpl) Size() int {
	return c.lruCache.Len()
}

func (c *cacheImpl) reportHit() {
	if c.reporter == nil {
		return
	}
	c.reporter.Report("BoundedCacheHit", 1, telemetry.UnitCount)
	// Reporting 0 to allow the use of Average to calculate the miss ratio
	c.reporter.Report("BoundedCacheMiss", 0, telemetry.UnitCount)

	c.reportSize()
}

func (c *cacheImpl) reportMiss() {
	if c.reporter == nil {
		return
	}

	c.reporter.Report("BoundedCacheMiss", 1, telemetry.UnitCount)
	// Reporting 0 to allow the use of Average to calculate the hit ratio
	c.reporter.Report("BoundedCacheHit", 0, telemetry.UnitCount)

	c.reportSize()
}

func (c *cacheImpl) reportSize() {
	if c.reporter == nil {
		return
	}

	// The average of this metric is the approximate size of the cache
	c.reporter.Report("BoundedCacheSize", float64(c.Size()), telemetry.UnitCount)
}
