package cache_test

import (
	"testing"
	"time"

	cache "code.justin.tv/amzn/TwitchBoundedCache"
	telemetry "code.justin.tv/amzn/TwitchTelemetry"
	"github.com/stretchr/testify/assert"
)

func TestCache_GetNotFound(t *testing.T) {
	c, err := cache.NewCache(cache.Config{})
	assert.NoError(t, err)

	val, ok := c.Get("a")
	assert.False(t, ok)
	assert.Nil(t, val)
}

func TestCache_SetAndGet(t *testing.T) {
	mockReporter, mockObserver := mockReporter()
	c, err := cache.NewCache(cache.Config{
		CacheSize: 3,
		MetricReporter: mockReporter,
	})
	assert.NoError(t, err)

	evicted := c.SetWithExpiration("a", "vala", 2*time.Second)
	assert.False(t, evicted)

	evicted = c.Set("b", "valb")
	assert.False(t, evicted)

	val, e, ok := c.GetWithExpiration("a")
	assert.True(t, ok)
	assert.Equal(t, "vala", val.(string))
	assert.False(t, e.IsZero())

	currentSize := c.Size()
	assert.Equal(t, 2, currentSize)

	assert.Equal(t, 3, len(mockObserver.Samples))
}

func TestCache_ExpireItem(t *testing.T) {
	mockReporter, mockObserver := mockReporter()
	c, err := cache.NewCache(cache.Config{
		CacheSize: 3,
		MetricReporter: mockReporter,
	})
	assert.NoError(t, err)

	evicted := c.SetWithExpiration("a", "vala", 2*time.Second)
	assert.False(t, evicted)

	time.Sleep(2 * time.Second)
	// a should expire now
	val, ok := c.Get("a")
	assert.False(t, ok)
	assert.Nil(t, val)

	assert.Equal(t, 3, len(mockObserver.Samples))
}

func TestCache_EvictItem(t *testing.T) {
	evictedMap := make(map[string]interface{})

	mockReporter, mockObserver := mockReporter()
	c, err := cache.NewCache(cache.Config{
		CacheSize: 2,
		OnEvicted: func(key string, val interface{}) {
			evictedMap[key] = val
		},
		MetricReporter: mockReporter,
	})
	assert.NoError(t, err)

	evicted := c.Set("a", "vala")
	assert.False(t, evicted)

	evicted = c.Set("b", "valb")
	assert.False(t, evicted)

	val, e, ok := c.GetWithExpiration("a")
	assert.True(t, ok)
	assert.Equal(t, "vala", val.(string))
	assert.True(t, e.IsZero())

	// b should be evicted
	evicted = c.Set("c", "valc")
	assert.True(t, evicted)

	val, ok = c.Get("b")
	assert.False(t, ok)
	assert.Nil(t, val)
	assert.Len(t, evictedMap, 1)
	assert.Contains(t, evictedMap, "b")

	assert.Equal(t, 6, len(mockObserver.Samples))
}

func TestCache_DeleteItem(t *testing.T) {
	evictedMap := make(map[string]interface{})

	mockReporter, mockObserver := mockReporter()
	c, err := cache.NewCache(cache.Config{
		CacheSize: 2,
		OnEvicted: func(key string, val interface{}) {
			evictedMap[key] = val
		},
		MetricReporter: mockReporter,
	})
	assert.NoError(t, err)

	evicted := c.SetWithExpiration("d", 1, 10*time.Second)
	assert.False(t, evicted)

	val, e, ok := c.PeekWithExpiration("d")
	assert.True(t, ok)
	assert.False(t, e.IsZero())
	assert.Equal(t, 1, val.(int))

	c.Delete("d")
	val, ok = c.Peek("d")
	assert.Nil(t, val)
	assert.False(t, ok)
	assert.Len(t, evictedMap, 1)
	assert.Contains(t, evictedMap, "d")

	assert.Equal(t, 6, len(mockObserver.Samples))
}

// Returns a SampleReporter and its fake SampleObserver that
// keeps track of the number of times it has been invoked.
func mockReporter() (*telemetry.SampleReporter, *mockSampleObserver) {
	reporter := &telemetry.SampleReporter{}
	reporter.ProcessIdentifier.Service = "TestService"
	mockObserver := &mockSampleObserver{}
	reporter.SampleObserver = mockObserver
	return reporter, mockObserver
}

// Taken from https://code.amazon.com/packages/TwitchTelemetry/blobs/2fd537a4f7da18ca5034a202591c11b719811151/--/sample_reporter_test.go#L85-L108
type mockSampleObserver struct {
	Samples []*telemetry.Sample
}

func (observer *mockSampleObserver) ObserveSample(sample *telemetry.Sample) {
	observer.Samples = append(observer.Samples, sample)
}

func (observer *mockSampleObserver) Flush() {
}

func (observer *mockSampleObserver) Stop() {
}

func (observer *mockSampleObserver) BufferedFlush(distributions []*telemetry.Distribution) {
}

func (observer *mockSampleObserver) ToMap() map[string]float64 {
	metrics := make(map[string]float64)
	for _, sample := range observer.Samples {
		metrics[sample.MetricID.Name] = sample.Value
	}
	return metrics
}
