package cache

import (
	"sync/atomic"
	"time"
)

type Clock interface {
	// the function return the nanoseconds elisped since January 1, 1970 UTC
	GetCurrentTime() int64
}

type systemClock struct{}

func (sysclock systemClock) GetCurrentTime() int64 {
	return time.Now().UnixNano()
}

type CachedClock struct {
	now    int64
	ticker *time.Ticker
	done   chan struct{}
}

func (cachedclock *CachedClock) GetCurrentTime() int64 {
	return atomic.LoadInt64(&cachedclock.now)
}

func (cachedclock *CachedClock) update() {
	for {
		select {
		case <-cachedclock.done:
			return
		case t := <-cachedclock.ticker.C:
			atomic.StoreInt64(&cachedclock.now, t.UnixNano())
		}
	}
}

// Stop will stop the goroutine that updates the time in the background
func (cachedclock *CachedClock) Stop() {
	cachedclock.ticker.Stop()
	close(cachedclock.done)
}

// CachedClock should always be created through NewCachedClock function instead of directly creating an instance of the struct
// this function will spin up the goroutine that update times automatically
func NewCachedClock() *CachedClock {
	clock := &CachedClock{
		now:    time.Now().UnixNano(),
		ticker: time.NewTicker(time.Second),
		done:   make(chan struct{}),
	}

	go clock.update()

	return clock
}
