# Email Validation Service (EVS)

EVS maintains email validation status for email addresses. 

# Quick Links

* [Runbook](docs/runbook.md) 
* [Threat Model](https://docs.google.com/document/d/1xUAt1q7AGmzCYhgShiDtCuItClE0LQHD1J5nNMqA8os/edit)
* [Grafana Dashboard](https://grafana.xarth.tv/d/000001420/growth-email-validator)
* [Rollbar](https://rollbar.com/Twitch/EmailValidator/)
* [Prod Logs](https://tiny.amazon.com/1b2ei8qz2/IsenLink)
* [Stage Logs](https://tiny.amazon.com/wyqtabjz/IsenLink)
* [twitch-email-validator Isengard](https://tiny.amazon.com/17jow6dm1/IsenLink)
* [twitch-email-validator+dev Isengard](https://tiny.amazon.com/ll1n17le/IsenLink)
* [Prod terraform](https://git-aws.internal.justin.tv/growth/terraform/tree/master/twitch-growth-aws/emailvalidator)
* [Stage terraform](https://git-aws.internal.justin.tv/growth/terraform/tree/master/twitch-growth-dev/emailvalidator)

| Environment | API Endpoint (Privatelink required)
| --- | --- |
| Prod | https://us-west-2.prod.twitchemailvalidatorservice.s.twitch.a2z.com
| Stage | https://us-west-2.beta.twitchemailvalidatorservice.s.twitch.a2z.com

## High Level Design

Each email verification is internally associated with a unique triple:

1) A namespace string (e.g., "communities")
2) An identifier string, which is required to be unique within the namespace
3) An email address

A verification can be in one of three states: "Pending", "Verified", or "Rejected".

Additionally, a verification request generates a new, unique, opaque ID that can be used to refer to the new, pending verification.  This opaque ID will be included in the link that's mailed to the user for clicking, to avoid leaking any personal or internal service information in the URL structure.

## Verifying a new email

The easiest way to use EVS is to import its Go client and follow the examples below.

When your service receives a new email address that it will need to send mail to, you can trigger a verification request (with a custom message) as follows:

```go
package main

import (
	"context"
	"log"
	"net/http"

	"code.justin.tv/foundation/twitchclient"
	evs "code.justin.tv/amzn/TwitchEmailValidatorServiceTwirp"
)

func main() {
    evsClient := evs.NewTwitchEmailValidatorServiceProtobufClient(
        "https://us-west-2.prod.twitchemailvalidatorservice.s.twitch.a2z.com",
        twitchclient.NewHTTPClient(twitchclient.ClientConf{
            Host: "https://us-west-2.prod.twitchemailvalidatorservice.s.twitch.a2z.com",
        }),
    )

	ctx := context.Background()

	req, err := evsClient.AddVerificationRequest(
		ctx, 
		&evs.AddVerificationRequestInput{
			Namespace: "my namespace",
			Key: "key", // usually user_id
			Email: "test@example.com",
			Locale: "en", // use an appropriate locale for the user
			Purpose: "some human readable purpose",
			ShouldIncludeCode: true, // this should almost always be true
		},
    )
	if err != nil {
        // Handle the error
    }

	log.Printf("verification request created: %v", req)
}
```

Note that this function doesn't make any attempt to verify that the provided email is a well-formed email address.

Assuming everything goes well, this will send the provided email address a link to click, and code to verify their email.  

When the user clicks the provided link, they will be taken to a twitch.tv landing page that will confirm that their email address is verified.

You can call `AddVerificationRequest` more than once with the same parameters; if the link in the email from the first request hasn't been clicked on, the second request will invalidate the first (so that link will no longer work), and a new email will be sent.

## Current Use Cases 2/22/2020

### GraphQL
Most of EVS traffic comes from GraphQL which attempts to fetch, resend, and verify verification requests. 

### pushy

New verification requests will send a verification request event to SNS. pushy subscribes to these events through SQS, and when it receives an event, it sends an email verification request to the email.

### developers (twitch-rbac)

Companies can register as developers on Twitch to do things like manage games and create Drops campaigns on Twitch.  
EVS is used to verify company emails matching a company's domain so that emails to developers can be sent to their company email. 

### extensions (visage)

When new extensions are added (see the `add_extension` endpoint in visage), a verification email request is sent to the author and support emails. These
e-mails are not specifically tied to any Twitch users.

When pushy attempts to send extension e-mails (see the extensions handler in pushy), it checks the verification status of the e-mail in the extensions namespace.

### users-service

When users-service creates new users (see `logic/create_user.go` in users-service), a verification email request is sent to the user's e-mail.

When EVS verifies an e-mail address, it publishes a verification success event to SNS. users-service subscribes to these events through SQS; when it
receives an event, it will look up the user by ID, and set the user's `email_verified` field to true in SiteDB.

If the `include_verification_code` parameter is set to true for a new user, a 6-digit verification code is generated for the convenience of mobile users and included in the Pushy email.
In addition to the opaque ID method, this code can also be used to verify email through the `verify_code` endpoint, or regenerated with the `regenerate_code` endpoint, which will send out a new email.


### Growth Debugger lambda
Used to fetch email verification status and Unreject verifications (for support)

### Delete User Data lambda
https://git-aws.internal.justin.tv/growth/delete-user-data calls into EVS to delete user data to comply with GDPR. 

# Development

See the [Fulton docs](https://docs.fulton.twitch.a2z.com/docs/course_joining_fulton_service.html) for instructions on how to get started and do testing. 

It is recommended to `export NAME=TwitchEmailValidatorService` prior to going through the Fulton setup.

## Jumpboxes:
Beta/Staging: `TC=twitch-emailvalidatorservice-beta-us-west-2 ssh jumpbox.fulton`

Prod: `TC=twitch-emailvalidatorservice-prod-us-west-2 ssh jumpbox.fulton`


## Generating Mocks
Add the interface that you would like to mock under the `mocks` command within the `Makefile`(examples are there for both first party and third party interfaces).
Run `brazil-build mocks` to generate the mocks. This is analogous to running `make mocks`, while also running boilerplate brazil setup.