package clients

import (
	"log"
	"net"
	"net/http"
	"time"

	"github.com/aws/aws-sdk-go/aws/endpoints"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sts"
)

func NewAssumedRoleSession(region string, roleARN string, httpClient *http.Client) (*session.Session, error) {
	stsSession, serr := session.NewSession(&aws.Config{
		Region:              aws.String(region),
		HTTPClient:          httpClient,
		STSRegionalEndpoint: endpoints.RegionalSTSEndpoint,
	})
	if serr != nil {
		return &session.Session{}, serr
	}
	stsClient := sts.New(stsSession)
	creds := stscreds.NewCredentialsWithClient(stsClient, roleARN)

	awsConfig := aws.NewConfig().
		WithMaxRetries(5).
		WithRegion(region).
		WithCredentials(creds).
		WithHTTPClient(httpClient)

	return session.NewSession(awsConfig)
}

func CreateHTTPClient(timeout time.Duration) *http.Client {
	if timeout < 500*time.Millisecond && timeout > 0 {
		log.Fatalf("timeout of %v is too low", timeout)
	}

	return &http.Client{
		Timeout: timeout,
		Transport: &http.Transport{
			DialContext: (&net.Dialer{
				Timeout:   30 * time.Second,
				KeepAlive: 30 * time.Second,
				DualStack: true,
			}).DialContext,
			MaxIdleConnsPerHost:   1000,
			IdleConnTimeout:       90 * time.Second,
			TLSHandshakeTimeout:   10 * time.Second,
			ExpectContinueTimeout: 1 * time.Second,
		},
	}
}
