package clients

import (
	"context"
	"errors"
	"fmt"
	"net/http"
)

// https://git-aws.internal.justin.tv/web/users-service/blob/master/doc/api.md#parameters
const DeleteFormat = "%s/users/%s?admin=email_validator_service&description=User%%20Selected%%20not%%20me%%20for%%20email%%20verification&isUserRequestedDestroy=true&flagged_to_delete=true"

type UserWrapper interface {
	EventualHardDeleteUser(ctx context.Context, userID string) error
}

func NewUserWrapper(baseURL string, httpClient *http.Client) (UserWrapper, error) {
	if baseURL == "" {
		return nil, errors.New("empty baseURL")
	}

	if httpClient == nil {
		return nil, errors.New("nil http client")
	}

	return &userWrapper{
		baseURL:    baseURL,
		httpClient: httpClient,
	}, nil
}

type userWrapper struct {
	baseURL    string
	httpClient *http.Client
}

func (c *userWrapper) EventualHardDeleteUser(ctx context.Context, userID string) error {
	if userID == "" {
		return errors.New("missing userID")
	}

	deleteURL := fmt.Sprintf(DeleteFormat, c.baseURL, userID)
	req, err := http.NewRequest("DELETE", deleteURL, nil)
	if err != nil {
		return errors.New(err.Error())
	}

	resp, err := c.httpClient.Do(req)
	if resp != nil {
		defer func() {
			if err2 := resp.Body.Close(); err == nil && err2 != nil {
				err = errors.New(err2.Error())
			}
		}()
	}

	return err
}
