package main

import (
	"context"
	"testing"

	"code.justin.tv/amzn/TwitchEmailValidatorService/models"

	serviceapi "code.justin.tv/amzn/TwitchEmailValidatorServiceTwirp"
	uuid "github.com/satori/go.uuid"
	"github.com/stretchr/testify/require"
	"github.com/twitchtv/twirp"
)

func TestIntegrationRegenerateCode(t *testing.T) {
	ctx := context.Background()
	client := protoClient(t)

	key := uuid.NewV4().String()
	req := createRequestWithKey(t, key)

	var err error

	newReq, err := client.RegenerateCode(ctx, &serviceapi.RegenerateCodeInput{
		Namespace: req.Namespace,
		Key:       key,
		Email:     req.Email,
	})
	require.NoError(t, err)
	require.Equal(t, models.StatusPendingString, newReq.VerificationRequest.Status)
	require.Equal(t, req.OpaqueID, newReq.VerificationRequest.OpaqueID)
	require.NotEqual(t, req.VerificationCode, newReq.VerificationRequest.VerificationCode)
	require.NotEqual(t, req.Modified, newReq.VerificationRequest.Modified)

	// Old code doesn't work
	_, err = client.VerifyCode(ctx, &serviceapi.VerifyCodeInput{
		Namespace: req.Namespace,
		Key:       req.Key,
		Email:     req.Email,
		Code:      req.VerificationCode,
	})
	requireTwirpErr(t, err, twirp.Unauthenticated)

	// Test that new code successfully verifies
	_, err = client.VerifyCode(ctx, &serviceapi.VerifyCodeInput{
		Namespace: newReq.VerificationRequest.Namespace,
		Key:       newReq.VerificationRequest.Key,
		Email:     newReq.VerificationRequest.Email,
		Code:      newReq.VerificationRequest.VerificationCode,
	})
	require.NoError(t, err)

	getOut, err := client.GetVerificationRequest(ctx, &serviceapi.GetVerificationRequestInput{
		Namespace: newReq.VerificationRequest.Namespace,
		Key:       newReq.VerificationRequest.Key,
		Email:     newReq.VerificationRequest.Email,
	})
	require.NoError(t, err)
	require.Equal(t, models.StatusVerifiedString, getOut.VerificationRequest.Status)

	// Regenerating code after successful verification fails
	_, err = client.RegenerateCode(ctx, &serviceapi.RegenerateCodeInput{
		Namespace: req.Namespace,
		Key:       req.Key,
		Email:     req.Email,
	})
	requireTwirpErr(t, err, twirp.FailedPrecondition)
}

func TestIntegrationRegenerateCodeMultipleTimes(t *testing.T) {
	ctx := context.Background()
	client := protoClient(t)

	key := uuid.NewV4().String()
	req := createRequestWithKey(t, key)

	for i := 0; i < 10; i++ {
		newReq, err := client.RegenerateCode(ctx, &serviceapi.RegenerateCodeInput{
			Namespace: req.Namespace,
			Key:       key,
			Email:     req.Email,
		})
		require.NoError(t, err)
		require.Equal(t, models.StatusPendingString, newReq.VerificationRequest.Status)
		require.Equal(t, req.OpaqueID, newReq.VerificationRequest.OpaqueID)
		require.NotEqual(t, req.VerificationCode, newReq.VerificationRequest.VerificationCode)

		req = newReq.VerificationRequest
	}
}
