package main

// See https://docs.fulton.twitch.a2z.com/docs/integration_tests.html for more information on how integration tests work
import (
	"context"
	"log"
	"os"
	"testing"

	"code.justin.tv/amzn/C7-go/c7"
	"code.justin.tv/amzn/C7-go/resolvers"
	"code.justin.tv/amzn/TwirpGoLangClientAutowire/c7autowire"
	"code.justin.tv/amzn/TwirpGoLangClientAutowire/configs"
	"code.justin.tv/amzn/TwitchEmailValidatorService/testutils"
	serviceapi "code.justin.tv/amzn/TwitchEmailValidatorServiceTwirp"
	"github.com/stretchr/testify/require"
	"github.com/twitchtv/twirp"
)

var (
	region    string
	stage     string
	namespace string
	email     string
)

// TestMain test setup.
func TestMain(m *testing.M) {
	if os.Getenv("HYDRA") == "" {
		// Noop since this means we are likely trying to run integration tests as part of unit testing
		log.Println("Not running integration tests under HYDRA, terminate")
		os.Exit(0)
	}

	stage = "beta"
	if os.Getenv("LOGICAL_STAGE_NAME") != "" {
		stage = os.Getenv("LOGICAL_STAGE_NAME")
	}

	namespace = testutils.RandStr(10)
	email = testutils.RandEmail()

	retCode := m.Run()

	os.Exit(retCode)
}

func jsonClient(t *testing.T) serviceapi.TwitchEmailValidatorService {
	t.Helper()
	lc := autowiredClient(t)

	// Get the twirp client
	return serviceapi.NewTwitchEmailValidatorServiceJSONClient(getClientEndpoint(t), lc)
}

func protoClient(t *testing.T) serviceapi.TwitchEmailValidatorService {
	t.Helper()
	lc := autowiredClient(t)
	return serviceapi.NewTwitchEmailValidatorServiceProtobufClient(getClientEndpoint(t), lc)
}

func autowiredClient(t *testing.T) serviceapi.HTTPClient {
	t.Helper()

	twirpAutowire := c7autowire.NewC7TwirpAutowire(loadC7(t))

	client, err := twirpAutowire.ClientForService("TwitchEmailValidatorServiceClient")
	require.NoError(t, err)
	return client
}

func loadC7(t *testing.T) *c7.C7 {
	t.Helper()

	configFile := "all.c7"
	if os.Getenv("FULTON_CONFIG") != "" {
		configFile = os.Getenv("FULTON_CONFIG")
	}

	set, err := resolvers.ResolveFile(configFile)
	require.NoError(t, err)

	return c7.NewC7(*set, region, stage)
}

func getClientEndpoint(t *testing.T) string {
	t.Helper()
	config := configs.TwirpClientConfig{}
	err := loadC7(t).FillWithNamespace("TwitchEmailValidatorServiceClient", &config)
	require.NoError(t, err)
	return config.Endpoint
}

func createRequestWithKey(t *testing.T, key string) *serviceapi.VerificationRequestDocument {
	ctx := context.Background()
	client := protoClient(t)

	out, err := client.AddVerificationRequest(ctx,
		&serviceapi.AddVerificationRequestInput{
			Namespace:         namespace,
			Key:               key,
			Email:             email,
			Locale:            "en",
			Purpose:           "purpose",
			ShouldIncludeCode: true,
		},
	)
	require.NoError(t, err)
	return out.VerificationRequest
}

func createRequestWithNamespaceAndKey(t *testing.T, namespace string, key string) *serviceapi.VerificationRequestDocument {
	ctx := context.Background()
	client := protoClient(t)
	out, err := client.AddVerificationRequest(ctx,
		&serviceapi.AddVerificationRequestInput{
			Namespace:         namespace,
			Key:               key,
			Email:             email,
			Locale:            "en",
			Purpose:           "purpose",
			ShouldIncludeCode: true,
		},
	)
	require.NoError(t, err)
	return out.VerificationRequest
}

func requireTwirpErr(t *testing.T, err error, code twirp.ErrorCode) {
	require.Error(t, err)
	twerr, ok := err.(twirp.Error)
	require.True(t, ok)
	require.Equal(t, twerr.Code(), code)
}
