package main

import (
	"context"
	"testing"

	"code.justin.tv/amzn/TwitchEmailValidatorService/models"
	datasource "code.justin.tv/amzn/TwitchEmailValidatorService/storage"
	serviceapi "code.justin.tv/amzn/TwitchEmailValidatorServiceTwirp"
	uuid "github.com/satori/go.uuid"
	"github.com/stretchr/testify/require"
	"github.com/twitchtv/twirp"
)

func TestIntegrationVerifyCode(t *testing.T) {
	ctx := context.Background()
	client := protoClient(t)

	namespace := uuid.NewV4().String()
	key := uuid.NewV4().String()

	req := createRequestWithNamespaceAndKey(t, namespace, key)

	verifyOut, err := client.VerifyCode(ctx, &serviceapi.VerifyCodeInput{
		Namespace: namespace,
		Key:       key,
		Email:     email,
		Code:      req.VerificationCode,
	})
	require.NoError(t, err)
	require.Equal(t, req.Namespace, verifyOut.VerificationRequest.Namespace)
	require.Equal(t, req.Email, verifyOut.VerificationRequest.Email)
	require.Equal(t, req.Key, verifyOut.VerificationRequest.Key)

	opaqueOut1, err := client.GetVerificationRequestByOpaqueID(ctx, &serviceapi.GetVerificationRequestByOpaqueIDInput{
		OpaqueId: req.OpaqueID,
	})
	require.NoError(t, err)
	require.Equal(t, models.StatusVerifiedString, opaqueOut1.VerificationRequest.Status)

	_, err = client.AddVerificationRequest(ctx,
		&serviceapi.AddVerificationRequestInput{
			Namespace:         namespace,
			Key:               key,
			Email:             email,
			Locale:            "en",
			Purpose:           "purpose",
			ShouldIncludeCode: true,
		},
	)
	requireTwirpErr(t, err, twirp.AlreadyExists)
}

func TestIntegrationVerifyCodeFailedAttempts(t *testing.T) {
	ctx := context.Background()
	client := protoClient(t)

	namespace := uuid.NewV4().String()
	key := uuid.NewV4().String()

	req := createRequestWithNamespaceAndKey(t, namespace, key)

	// Keep getting it wrong
	for i := 0; i < datasource.MaxVerificationCodeAttempts-1; i++ {
		_, err := client.VerifyCode(ctx, &serviceapi.VerifyCodeInput{
			Namespace: namespace,
			Key:       key,
			Email:     email,
			Code:      "RONG",
		})
		requireTwirpErr(t, err, twirp.Unauthenticated)
	}

	// Hit the limit to trigger a regenerate, and get back an appropriate error message
	_, err := client.VerifyCode(ctx, &serviceapi.VerifyCodeInput{
		Namespace: namespace,
		Key:       key,
		Email:     email,
		Code:      "RONG",
	})
	requireTwirpErr(t, err, twirp.ResourceExhausted)

	newRequest, err := client.GetVerificationRequestByOpaqueID(ctx, &serviceapi.GetVerificationRequestByOpaqueIDInput{
		OpaqueId: req.OpaqueID,
	})
	require.NoError(t, err)
	require.Equal(t, models.StatusPendingString, newRequest.VerificationRequest.Status)
	require.NotEqual(t, req.Modified, newRequest.VerificationRequest.Modified)

	// Try verifying with the old code and fail
	_, err = client.VerifyCode(ctx, &serviceapi.VerifyCodeInput{
		Namespace: namespace,
		Key:       key,
		Email:     email,
		Code:      req.VerificationCode,
	})
	requireTwirpErr(t, err, twirp.Unauthenticated)

	// Try verifying with the new code and succeed
	_, err = client.VerifyCode(ctx, &serviceapi.VerifyCodeInput{
		Namespace: namespace,
		Key:       key,
		Email:     email,
		Code:      newRequest.VerificationRequest.VerificationCode,
	})
	require.NoError(t, err)

	newRequest, err = client.GetVerificationRequestByOpaqueID(ctx, &serviceapi.GetVerificationRequestByOpaqueIDInput{
		OpaqueId: req.OpaqueID,
	})
	require.NoError(t, err)
	require.Equal(t, models.StatusVerifiedString, newRequest.VerificationRequest.Status)
}

func TestIntegrationVerifyCodeForCodelessVerification(t *testing.T) {
	ctx := context.Background()
	client := protoClient(t)

	namespace := uuid.NewV4().String()
	key := uuid.NewV4().String()

	out, err := client.AddVerificationRequest(ctx,
		&serviceapi.AddVerificationRequestInput{
			Namespace:         namespace,
			Key:               key,
			Email:             email,
			Locale:            "en",
			Purpose:           "purpose",
			ShouldIncludeCode: false,
		},
	)
	require.NoError(t, err)

	_, err = client.VerifyCode(ctx, &serviceapi.VerifyCodeInput{
		Namespace: namespace,
		Key:       key,
		Email:     email,
		Code:      "RONG",
	})
	requireTwirpErr(t, err, twirp.FailedPrecondition)

	req, err := client.GetVerificationRequestByOpaqueID(ctx, &serviceapi.GetVerificationRequestByOpaqueIDInput{
		OpaqueId: out.VerificationRequest.OpaqueID,
	})
	require.NoError(t, err)
	require.Equal(t, models.StatusPendingString, req.VerificationRequest.Status)
}

func TestIntegrationVerifyCodeNotFound(t *testing.T) {
	ctx := context.Background()
	client := protoClient(t)

	_, err := client.VerifyCode(ctx, &serviceapi.VerifyCodeInput{
		Namespace: uuid.NewV4().String(),
		Key:       uuid.NewV4().String(),
		Email:     email,
		Code:      "RONG",
	})
	requireTwirpErr(t, err, twirp.NotFound)
}
