package models

import (
	"crypto/rand"
	"fmt"
	"math/big"
)

const (
	// StatusPendingString represents the pending status
	StatusPendingString = "Pending"
	// StatusVerifiedString represents the verified status
	StatusVerifiedString = "Verified"
	// StatusRejectedString represents the rejected status
	StatusRejectedString = "Rejected"

	// OpaqueIDLength is the length of the generated opaque ID
	OpaqueIDLength = 32

	// VerificationCodeLength is the length of the verification code for mobile devices
	VerificationCodeLength = 6
)

var (
	ValidOpaqueIDChars         = []byte("abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789")
	ValidVerificationCodeChars = []byte("0123456789")
)

// MakeCompoundKey builds the triple-compound key for dynamo
func MakeCompoundKey(namespace, id, email string) string {
	return namespace + "##" + id + "##" + email
}

// MakeOpaqueID provides a randomized opaque identifier.  It's used in the
// email link to verify a namespace, user ID, and email triplet
func MakeOpaqueID() (string, error) {
	return generateSecureCode(ValidOpaqueIDChars, OpaqueIDLength)
}

// MakeVerificationCode generates a random numeric verification code for the
// convenience of mobile devices
func MakeVerificationCode() (string, error) {
	return generateSecureCode(ValidVerificationCodeChars, VerificationCodeLength)
}

func generateSecureCode(validChars []byte, length int) (string, error) {
	codeBytes := make([]byte, length)
	for i := 0; i < length; i++ {
		num, err := rand.Int(rand.Reader, big.NewInt(int64(len(validChars))))
		if err != nil {
			return "", err
		}

		n := num.Int64()
		codeBytes[i] = validChars[n]
	}

	return string(codeBytes), nil
}

// StatusString provides a human-readable representation of an
// email verification state.
func StatusString(state VerificationStatus) string {
	switch state {
	case StatusPending:
		return StatusPendingString
	case StatusVerified:
		return StatusVerifiedString
	case StatusRejected:
		return StatusRejectedString
	default:
		return fmt.Sprintf("Unknown verification state: %v", state)
	}
}
