package publisher

import (
	"context"
	"encoding/json"
	"log"

	"code.justin.tv/amzn/TwitchEmailValidatorService/evs/documents"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/sns"
	"github.com/aws/aws-sdk-go/service/sns/snsiface"
)

const (
	xactGroup        = "sns"
	origin           = "growth/emailvalidator"
	requestEventName = "emailvalidationrequest"
	successEventName = "emailvalidationsuccess"
)

// Publisher is the interface that encapsulates our connection to
// pushy for sending email verification requests
type Publisher interface {
	PublishVerificationRequest(ctx context.Context, email, opaqueID, locale, purpose string, recipientID string, verificationCode string) error
	PublishVerificationSuccess(ctx context.Context, namespace, key, email, locale string) error
}

type snsPublisher struct {
	sns      snsiface.SNSAPI
	topicARN string
}

// NewPublisher constructs an SNS publisher
func NewSNSPublisher(svc snsiface.SNSAPI, topicARN string) Publisher {
	return &snsPublisher{
		sns:      svc,
		topicARN: topicARN,
	}
}

// PublishVerificationRequest sends the actual request to pushy
func (p *snsPublisher) PublishVerificationRequest(ctx context.Context, email, opaqueID, locale, purpose string, recipientID string, verificationCode string) error {
	data := documents.EmailValidationEventParams{
		Email:            email,
		OpaqueID:         opaqueID,
		Purpose:          purpose,
		Locale:           locale,
		RecipientID:      recipientID,
		VerificationCode: verificationCode,
	}

	bytes, err := json.Marshal(data)
	if err != nil {
		return err
	}

	in := &sns.PublishInput{
		Message: aws.String(string(bytes)),
		MessageAttributes: map[string]*sns.MessageAttributeValue{
			"event": {
				StringValue: aws.String(requestEventName),
				DataType:    aws.String("String"),
			},
			"origin": {
				StringValue: aws.String(origin),
				DataType:    aws.String("String"),
			},
		},
		TopicArn: aws.String(p.topicARN),
	}
	log.Printf("Posting request message to SNS (opaque ID: %s)", opaqueID)
	_, err = p.sns.PublishWithContext(ctx, in)
	return err
}

// PublishVerification sends information about a successful verification to SNS
func (p *snsPublisher) PublishVerificationSuccess(ctx context.Context, namespace, key, email, locale string) error {
	data := documents.EmailValidationSuccessEventParams{
		Namespace: namespace,
		Key:       key,
		Email:     email,
		Locale:    locale,
	}

	bytes, err := json.Marshal(data)
	if err != nil {
		return err
	}

	in := &sns.PublishInput{
		Message: aws.String(string(bytes)),
		MessageAttributes: map[string]*sns.MessageAttributeValue{
			"event": {
				StringValue: aws.String(successEventName),
				DataType:    aws.String("String"),
			},
			"origin": {
				StringValue: aws.String(origin),
				DataType:    aws.String("String"),
			},
		},
		TopicArn: aws.String(p.topicARN),
	}
	log.Printf("Posting success message to SNS (namespace: %s, key: %s, locale %s)", namespace, key, locale)
	_, err = p.sns.PublishWithContext(ctx, in)
	return err
}
