package server_test

import (
	"context"

	"code.justin.tv/amzn/TwitchEmailValidatorService/server/converter"
	"code.justin.tv/amzn/TwitchEmailValidatorService/testutils"
	evs "code.justin.tv/amzn/TwitchEmailValidatorServiceTwirp"
	"github.com/pkg/errors"
	"github.com/stretchr/testify/mock"
	"github.com/twitchtv/twirp"
)

func (s *ServerSuite) TestAddVerificationRequestSuccess() {
	namespace := testutils.RandStr(10)
	key := testutils.RandStr(10)
	email := testutils.RandStr(10)
	locale := testutils.RandStr(2)
	purpose := testutils.RandStr(10)
	req := testutils.NewVerificationRequest()

	expectedVerificationRequest, err := converter.OutputVerificationRequest(req)
	s.Require().NoError(err)

	s.Mocks.DS.On("CreateVerificationRequest", mock.Anything, namespace, key, email, locale, false).Return(req, nil).Once()
	s.Mocks.PushyPublisher.On(
		"PublishVerificationRequest", mock.Anything, req.Email, req.OpaqueID, req.Locale, purpose, mock.Anything, req.VerificationCode,
	).Return(nil).Once()

	out, err := s.service.AddVerificationRequest(context.Background(), &evs.AddVerificationRequestInput{
		Namespace: namespace,
		Key:       key,
		Email:     email,
		Locale:    locale,
		Purpose:   purpose,
	})
	s.Require().NoError(err)
	s.Require().Equal(expectedVerificationRequest, out.VerificationRequest)
	s.AssertExpectations()
}

func (s *ServerSuite) TestAddVerificationRequestIncludeCodeSuccess() {
	namespace := testutils.RandStr(10)
	key := testutils.RandStr(10)
	email := testutils.RandStr(10)
	locale := testutils.RandStr(2)
	purpose := testutils.RandStr(10)
	shouldIncludeCode := true
	req := testutils.NewVerificationRequest()

	expectedVerificationRequest, err := converter.OutputVerificationRequest(req)
	s.Require().NoError(err)

	s.Mocks.DS.On("CreateVerificationRequest", mock.Anything, namespace, key, email, locale, shouldIncludeCode).Return(req, nil).Once()
	s.Mocks.PushyPublisher.On(
		"PublishVerificationRequest", mock.Anything, req.Email, req.OpaqueID, req.Locale, purpose, mock.Anything, req.VerificationCode,
	).Return(nil).Once()

	out, err := s.service.AddVerificationRequest(context.Background(), &evs.AddVerificationRequestInput{
		Namespace:         namespace,
		Key:               key,
		Email:             email,
		Locale:            locale,
		Purpose:           purpose,
		ShouldIncludeCode: shouldIncludeCode,
	})
	s.Require().NoError(err)
	s.Require().Equal(expectedVerificationRequest, out.VerificationRequest)
	s.AssertExpectations()
}

func (s *ServerSuite) TestAddVerificationRequestInvalid() {
	tests := []struct {
		mut func(input *evs.AddVerificationRequestInput)
	}{
		{func(input *evs.AddVerificationRequestInput) {
			input.Namespace = ""
		}},
		{func(input *evs.AddVerificationRequestInput) {
			input.Key = ""
		}},
		{func(input *evs.AddVerificationRequestInput) {
			input.Email = ""
		}},
		{func(input *evs.AddVerificationRequestInput) {
			input.Purpose = ""
		}},
	}

	for _, tt := range tests {
		input := &evs.AddVerificationRequestInput{
			Namespace: testutils.RandStr(10),
			Key:       testutils.RandStr(10),
			Email:     testutils.RandStr(10),
			Locale:    testutils.RandStr(2),
			Purpose:   testutils.RandStr(10),
		}
		tt.mut(input)

		_, err := s.service.AddVerificationRequest(context.Background(), input)
		s.Require().Error(err)
		s.AssertExpectations()

		twerr := err.(twirp.Error)
		s.Require().Equal(twirp.InvalidArgument, twerr.Code())
	}
}

func (s *ServerSuite) TestAddVerifcationRequestError() {
	namespace := testutils.RandStr(10)
	key := testutils.RandStr(10)
	email := testutils.RandStr(10)
	locale := testutils.RandStr(2)
	purpose := testutils.RandStr(10)
	s.Mocks.DS.On("CreateVerificationRequest", mock.Anything, namespace, key, email, locale, false).Return(nil, errors.New("o no")).Once()

	_, err := s.service.AddVerificationRequest(context.Background(), &evs.AddVerificationRequestInput{
		Namespace: namespace,
		Key:       key,
		Email:     email,
		Locale:    locale,
		Purpose:   purpose,
	})
	s.Require().Error(err)
}
