package server

import (
	"time"

	"code.justin.tv/video/metricmiddleware-beta/operation"

	logging "code.justin.tv/amzn/TwitchLogging"
	telemetry "code.justin.tv/amzn/TwitchTelemetry"
	"code.justin.tv/video/metricmiddleware-beta/awsmetric"
	"github.com/aws/aws-sdk-go/service/dynamodb"

	"code.justin.tv/amzn/TwitchEmailValidatorService/c7s"
	"code.justin.tv/amzn/TwitchEmailValidatorService/clients"
	"code.justin.tv/amzn/TwitchEmailValidatorService/publisher"
	"code.justin.tv/amzn/TwitchEmailValidatorService/storage"
	"github.com/aws/aws-sdk-go/service/sns"
	"github.com/pkg/errors"
)

// Clients stores interfaces used by EmailValidator API.
type Clients struct {
	DS                  storage.VerifierAPI
	PushySNSPublisher   publisher.Publisher
	SuccessSNSPublisher publisher.Publisher
	Reporter            telemetry.SampleReporter
}

// InitClients initializes all the clients needed for the api
func InitClients(conf c7s.Config, logger logging.Logger, starter *operation.Starter, reporter telemetry.SampleReporter) (Clients, error) {

	httpClient := clients.CreateHTTPClient(20 * time.Second)
	// Initialize dynamo with the appropriate role
	dynamoSession, err := clients.NewAssumedRoleSession(conf.DynamoDBRegion, conf.DynamoDBRoleARN, httpClient)
	if err != nil {
		return Clients{}, errors.Wrap(err, "initiating dynamo session")
	}

	awsMetricClient := &awsmetric.Client{Starter: starter}
	dynamoSession = awsMetricClient.AddToSession(dynamoSession)
	ddb := dynamodb.New(dynamoSession)

	user, err := clients.NewUserWrapper(conf.UsersServiceURL, httpClient)
	if err != nil {
		return Clients{}, errors.Wrap(err, "creating users service wrapper")
	}

	// Initialize the datastore with the dynamo session
	newDatasourceInput := storage.NewDatasourceInput{
		DB:               ddb,
		ValidationsTable: conf.DynamoDBValidationsTable,
		User:             user,
		Logger:           logger,
	}
	ds, err := storage.NewDatasource(newDatasourceInput)
	if err != nil {
		return Clients{}, errors.Wrap(err, "creating datasource")
	}

	// Initialize the SNS Publishers the with appropriate role
	snsSession, err := clients.NewAssumedRoleSession(conf.AWSRegion, conf.SNSRoleARN, httpClient)
	if err != nil {
		return Clients{}, errors.Wrap(err, "initiating sns session")
	}

	svc := sns.New(snsSession)
	pushySNSPublisher := publisher.NewSNSPublisher(svc, conf.SNSRequestARN)
	successSNSPublisher := publisher.NewSNSPublisher(svc, conf.SNSSuccessARN)

	return Clients{
		DS:                  ds,
		PushySNSPublisher:   pushySNSPublisher,
		SuccessSNSPublisher: successSNSPublisher,
		Reporter:            reporter,
	}, nil
}
