package server_test

import (
	"context"
	"errors"

	"code.justin.tv/amzn/TwitchEmailValidatorService/models"
	"code.justin.tv/amzn/TwitchEmailValidatorService/server/converter"
	"code.justin.tv/amzn/TwitchEmailValidatorService/testutils"
	evs "code.justin.tv/amzn/TwitchEmailValidatorServiceTwirp"
	"github.com/stretchr/testify/mock"
	"github.com/twitchtv/twirp"
)

func (s *ServerSuite) TestListVerificationRequestsByKeySuccess() {
	key := testutils.RandStr(10)
	contToken := testutils.RandStr(10)
	newContToken := testutils.RandStr(10)
	limit := uint32(5)
	input := evs.ListVerificationRequestsByKeyInput{
		Key:               key,
		ContinuationToken: contToken,
		Limit:             limit,
	}

	req1 := testutils.NewVerificationRequest()
	expectedVerificationRequest1, err := converter.OutputVerificationRequest(req1)
	s.Require().NoError(err)

	req2 := testutils.NewVerificationRequest()
	expectedVerificationRequest2, err := converter.OutputVerificationRequest(req2)
	s.Require().NoError(err)

	s.Mocks.DS.On("ListVerificationRequestsByKey", mock.Anything, key, models.ListVerificationsParams{
		ContinuationToken: contToken,
		Limit:             uint(limit),
	}).Return(&models.ListVerificationsOutput{
		VerificationRequests: []models.VerificationRequest{
			*req1,
			*req2,
		},
		ContinuationToken: newContToken,
	}, nil).Once()

	listOut, err := s.service.ListVerificationRequestsByKey(context.Background(), &input)
	s.Require().NoError(err)

	s.Require().Equal(expectedVerificationRequest1, listOut.VerificationRequests[0])
	s.Require().Equal(expectedVerificationRequest2, listOut.VerificationRequests[1])
	s.Require().Equal(newContToken, listOut.ContinuationToken)

	s.AssertExpectations()
}

func (s *ServerSuite) TestListVerificationRequestsByKeyInvalid() {
	input := &evs.ListVerificationRequestsByKeyInput{}

	_, err := s.service.ListVerificationRequestsByKey(context.Background(), input)
	s.Require().Error(err)
	s.AssertExpectations()

	twerr := err.(twirp.Error)
	s.Require().Equal(twirp.InvalidArgument, twerr.Code())
}

func (s *ServerSuite) TestListVerificationRequestsByKeyDeleteError() {
	key := testutils.RandStr(10)

	s.Mocks.DS.On("ListVerificationRequestsByKey", mock.Anything, key, mock.Anything).Return(nil, errors.New("can't sorry")).Once()

	_, err := s.service.ListVerificationRequestsByKey(context.Background(), &evs.ListVerificationRequestsByKeyInput{
		Key: key,
	})

	s.Require().Error(err)
	s.AssertExpectations()
}
