package server

import (
	"context"
	"log"
	"strings"
	"time"

	"code.justin.tv/amzn/TwitchEmailValidatorService/models"
	telemetry "code.justin.tv/amzn/TwitchTelemetry"

	"code.justin.tv/amzn/TwitchEmailValidatorService/server/converter"
	"code.justin.tv/amzn/TwitchEmailValidatorService/storage"
	"github.com/pkg/errors"

	"github.com/twitchtv/twirp"

	evs "code.justin.tv/amzn/TwitchEmailValidatorServiceTwirp"
)

const (
	AcceptLanguage = "Accept-Language"
	XForwardedFor  = "X-Forwarded-For"
)

func (s *TwitchEmailValidatorServiceServer) RegenerateCode(ctx context.Context, req *evs.RegenerateCodeInput) (*evs.RegenerateCodeOutput, error) {
	if req.GetEmail() == "" {
		return nil, twirp.RequiredArgumentError("email")
	}
	if req.GetKey() == "" {
		return nil, twirp.RequiredArgumentError("key")
	}

	namespace := req.GetNamespace()
	if namespace == "" {
		namespace = usersServiceNamespace
	}

	if forwardedFor, ok := ctx.Value(XForwardedFor).(string); ok {
		log.Printf("regen ip: %s", forwardedFor)
	}

	// Silently short circuit if email is in blacklist
	if isBlacklistedEmail(req.Email) {
		opaqueID, err := models.MakeOpaqueID()
		if err != nil {
			return nil, getTwirpError(err)
		}

		code, err := models.MakeVerificationCode()
		if err != nil {
			return nil, getTwirpError(err)
		}

		fakeReq := &models.VerificationRequest{
			Namespace:        namespace,
			Email:            req.GetEmail(),
			Key:              req.GetKey(),
			OpaqueID:         opaqueID,
			Status:           models.StatusPending,
			Modified:         time.Now(),
			Locale:           "en",
			VerificationCode: code,
		}

		verificationRequest, err := converter.OutputVerificationRequest(fakeReq)
		if err != nil {
			return nil, twirp.InternalErrorWith(errors.Wrap(err, "converting verification request"))
		}

		s.Clients.Reporter.Report("regeneratecode.blacklist", 1, telemetry.UnitCount)
		return &evs.RegenerateCodeOutput{VerificationRequest: verificationRequest}, nil
	}

	request, err := s.Clients.DS.RegenerateCode(ctx, namespace, req.GetKey(), req.GetEmail())
	if err != nil {
		// Code doesn't exist, so generate a new one, this is to support legacy accounts
		if err == storage.ErrValidationNotFound {
			var acceptLanguage string
			var ok bool
			if acceptLanguage, ok = ctx.Value(AcceptLanguage).(string); !ok || acceptLanguage == "" {
				acceptLanguage = "en"
			}

			request, err := s.Clients.DS.CreateVerificationRequest(ctx, namespace, req.Key, req.Email, acceptLanguage, true)
			if err != nil {
				return nil, getTwirpError(err)
			}

			s.publishVerificationRequest(request, "Requested verification code regeneration")

			verificationRequest, err := converter.OutputVerificationRequest(request)
			if err != nil {
				return nil, twirp.InternalErrorWith(errors.Wrap(err, "converting verification request"))
			}

			return &evs.RegenerateCodeOutput{VerificationRequest: verificationRequest}, nil
		}

		return nil, getTwirpError(err)
	}

	s.publishVerificationRequest(request, "Requested verification code regeneration")

	verificationRequest, err := converter.OutputVerificationRequest(request)
	if err != nil {
		return nil, twirp.InternalErrorWith(errors.Wrap(err, "converting verification request"))
	}

	return &evs.RegenerateCodeOutput{VerificationRequest: verificationRequest}, nil
}

var blacklistHandles = map[string]bool{
	"tylerdayson1141":  true,
	"jocelynbrown3085": true,
}

// Temporary blacklist as a result of INC-1222
func isBlacklistedEmail(email string) bool {
	parts := strings.Split(email, "@")
	if len(parts) != 2 || parts[1] != "gmail.com" {
		return false
	}

	// Strip out everything after +
	handle := strings.Split(parts[0], "+")[0]

	// Strip out periods
	handle = strings.Replace(handle, ".", "", -1)
	return blacklistHandles[handle]
}
