package server_test

import (
	"context"
	"fmt"

	"code.justin.tv/amzn/TwitchEmailValidatorService/server"
	"code.justin.tv/amzn/TwitchEmailValidatorService/storage"

	"code.justin.tv/amzn/TwitchEmailValidatorService/server/converter"
	"code.justin.tv/amzn/TwitchEmailValidatorService/testutils"
	evs "code.justin.tv/amzn/TwitchEmailValidatorServiceTwirp"
	"github.com/pkg/errors"
	"github.com/stretchr/testify/mock"
	"github.com/twitchtv/twirp"
)

func (s *ServerSuite) TestRegenerateCodeSuccess() {
	namespace := testutils.RandStr(10)
	key := testutils.RandStr(10)
	email := testutils.RandStr(10)
	req := testutils.NewVerificationRequest()

	expectedVerificationRequest, err := converter.OutputVerificationRequest(req)
	s.Require().NoError(err)

	s.Mocks.DS.On("RegenerateCode", mock.Anything, namespace, key, email).Return(req, nil).Once()
	s.Mocks.PushyPublisher.On(
		"PublishVerificationRequest", mock.Anything, req.Email, req.OpaqueID, req.Locale, "Requested verification code regeneration", mock.Anything, req.VerificationCode,
	).Return(nil).Once()

	out, err := s.service.RegenerateCode(context.Background(), &evs.RegenerateCodeInput{
		Namespace: namespace,
		Key:       key,
		Email:     email,
	})

	s.Require().NoError(err)
	s.Require().Equal(expectedVerificationRequest, out.VerificationRequest)
	s.AssertExpectations()
}

func (s *ServerSuite) TestRegenerateCodeNamespaceSuccess() {
	key := testutils.RandStr(10)
	email := testutils.RandStr(10)
	req := testutils.NewVerificationRequest()

	expectedVerificationRequest, err := converter.OutputVerificationRequest(req)
	s.Require().NoError(err)

	s.Mocks.DS.On("RegenerateCode", mock.Anything, "users-service", key, email).Return(req, nil).Once()
	s.Mocks.PushyPublisher.On(
		"PublishVerificationRequest", mock.Anything, req.Email, req.OpaqueID, req.Locale, "Requested verification code regeneration", mock.Anything, req.VerificationCode,
	).Return(nil).Once()

	out, err := s.service.RegenerateCode(context.Background(), &evs.RegenerateCodeInput{
		Key:   key,
		Email: email,
	})

	s.Require().NoError(err)
	s.Require().Equal(expectedVerificationRequest, out.VerificationRequest)
	s.AssertExpectations()
}

func (s *ServerSuite) TestRegenerateCodeSuccessRecreateIfNotFound() {
	key := testutils.RandStr(10)
	email := testutils.RandStr(10)
	req := testutils.NewVerificationRequest()
	expectedLocale := testutils.RandStr(2)

	expectedVerificationRequest, err := converter.OutputVerificationRequest(req)
	s.Require().NoError(err)

	s.Mocks.DS.On("RegenerateCode", mock.Anything, "users-service", key, email).Return(nil, storage.ErrValidationNotFound).Once()
	s.Mocks.DS.On("CreateVerificationRequest", mock.Anything, "users-service", key, email, expectedLocale, true).Return(req, nil).Once()
	s.Mocks.PushyPublisher.On(
		"PublishVerificationRequest", mock.Anything, req.Email, req.OpaqueID, req.Locale, "Requested verification code regeneration", mock.Anything, req.VerificationCode,
	).Return(nil).Once()

	ctx := context.Background()
	ctx = context.WithValue(ctx, server.AcceptLanguage, expectedLocale)
	out, err := s.service.RegenerateCode(ctx, &evs.RegenerateCodeInput{
		Key:   key,
		Email: email,
	})

	s.Require().NoError(err)
	s.Require().Equal(expectedVerificationRequest, out.VerificationRequest)
	s.AssertExpectations()
}

func (s *ServerSuite) TestRegenerateCodeFailureRecreateIfNotFound() {
	key := testutils.RandStr(10)
	email := testutils.RandStr(10)
	expectedLocale := testutils.RandStr(2)

	s.Mocks.DS.On("RegenerateCode", mock.Anything, "users-service", key, email).Return(nil, storage.ErrValidationNotFound).Once()
	s.Mocks.DS.On("CreateVerificationRequest", mock.Anything, "users-service", key, email, expectedLocale, true).Return(nil, fmt.Errorf("failed")).Once()

	ctx := context.Background()
	ctx = context.WithValue(ctx, server.AcceptLanguage, expectedLocale)
	_, err := s.service.RegenerateCode(ctx, &evs.RegenerateCodeInput{
		Key:   key,
		Email: email,
	})

	s.Require().Error(err)
	s.AssertExpectations()
}

func (s *ServerSuite) TestRegenerateCodeSuccessRecreateIfNotFoundMissingAcceptLanguageHeader() {
	key := testutils.RandStr(10)
	email := testutils.RandStr(10)
	req := testutils.NewVerificationRequest()
	expectedLocale := "en"

	expectedVerificationRequest, err := converter.OutputVerificationRequest(req)
	s.Require().NoError(err)

	s.Mocks.DS.On("RegenerateCode", mock.Anything, "users-service", key, email).Return(nil, storage.ErrValidationNotFound).Once()
	s.Mocks.DS.On("CreateVerificationRequest", mock.Anything, "users-service", key, email, expectedLocale, true).Return(req, nil).Once()
	s.Mocks.PushyPublisher.On(
		"PublishVerificationRequest", mock.Anything, req.Email, req.OpaqueID, req.Locale, "Requested verification code regeneration", mock.Anything, req.VerificationCode,
	).Return(nil).Once()

	out, err := s.service.RegenerateCode(context.Background(), &evs.RegenerateCodeInput{
		Key:   key,
		Email: email,
	})

	s.Require().NoError(err)
	s.Require().Equal(expectedVerificationRequest, out.VerificationRequest)
	s.AssertExpectations()
}

func (s *ServerSuite) TestRegenerateCodeInvalid() {
	tests := []struct {
		mut func(input *evs.RegenerateCodeInput)
	}{
		{func(input *evs.RegenerateCodeInput) {
			input.Key = ""
		}},
		{func(input *evs.RegenerateCodeInput) {
			input.Email = ""
		}},
	}

	for _, tt := range tests {
		input := &evs.RegenerateCodeInput{
			Namespace: testutils.RandStr(10),
			Key:       testutils.RandStr(10),
			Email:     testutils.RandStr(10),
		}
		tt.mut(input)

		_, err := s.service.RegenerateCode(context.Background(), input)
		s.Require().Error(err)
		s.AssertExpectations()

		twerr := err.(twirp.Error)
		s.Require().Equal(twirp.InvalidArgument, twerr.Code())
	}
}

func (s *ServerSuite) TestRegenerateCodeError() {
	namespace := testutils.RandStr(10)
	key := testutils.RandStr(10)
	email := testutils.RandStr(10)
	s.Mocks.DS.On("RegenerateCode", mock.Anything, namespace, key, email).Return(nil, errors.New("o no")).Once()

	_, err := s.service.RegenerateCode(context.Background(), &evs.RegenerateCodeInput{
		Namespace: namespace,
		Key:       key,
		Email:     email,
	})
	s.Require().Error(err)
}

func (s *ServerSuite) TestRegenerateCodeBlacklisted() {
	tests := []struct {
		mut func(input *evs.RegenerateCodeInput)
	}{
		{func(input *evs.RegenerateCodeInput) {
			input.Email = "tylerdayson1141@gmail.com"
		}},
		{func(input *evs.RegenerateCodeInput) {
			input.Email = "tyler.dayson1141+testeroo@gmail.com"
		}},
		{func(input *evs.RegenerateCodeInput) {
			input.Email = "tylerd.a.ys.on11.41@gmail.com"
		}},
		{func(input *evs.RegenerateCodeInput) {
			input.Email = "tylerd.ay.s.on1.1.4.1@gmail.com"
		}},
		{func(input *evs.RegenerateCodeInput) {
			input.Email = "tylerday.so.n.11.4.1@gmail.com"
		}},
		{func(input *evs.RegenerateCodeInput) {
			input.Email = "tylerdayson1.1.41@gmail.com"
		}},
	}

	for _, tt := range tests {
		input := &evs.RegenerateCodeInput{
			Namespace: testutils.RandStr(10),
			Key:       testutils.RandStr(10),
			Email:     testutils.RandStr(10),
		}
		tt.mut(input)

		_, err := s.service.RegenerateCode(context.Background(), input)
		s.Require().NoError(err)
		s.AssertExpectations()
	}
}
