package server

// The implementation of a Twirp server
// See https://twitchtv.github.io/twirp/docs/example.html#implement-the-server
// We generate your Twirp server interfaces in TwitchEmailValidatorServiceTwirp
// by taking a dependency on TwitchEmailValidatorServiceSchema

import (
	"context"
	"errors"
	"log"
	"time"

	"code.justin.tv/amzn/TwitchEmailValidatorService/models"
	"code.justin.tv/amzn/TwitchEmailValidatorService/storage"
	logging "code.justin.tv/amzn/TwitchLogging"
	telemetry "code.justin.tv/amzn/TwitchTelemetry"
	"code.justin.tv/chat/golibs/logx"
	"github.com/twitchtv/twirp"
)

// TwitchEmailValidatorServiceServer An implementation of the TwitchEmailValidatorServiceServer Twirp service.
type TwitchEmailValidatorServiceServer struct {
	sampleReporter telemetry.SampleReporter
	logger         logging.Logger
	Clients        Clients
}

const (
	usersServiceNamespace = "users-service"
	handlerTimeout        = 20 * time.Second
)

var (
	isColdStart = true

	// ErrBodyDecode is returned when a request JSON body fails to decode
	ErrBodyDecode = errors.New("Couldn't decode request body")
)

// NewServer creates a new TwitchEmailValidatorServiceServer. Expects to be executed
// inside an AWS ECS execution environment.
func NewServer(sampleReporter telemetry.SampleReporter, logger logging.Logger, clients Clients) (*TwitchEmailValidatorServiceServer, error) {
	// Report a cold start on every new server
	if isColdStart {
		sampleReporter.Report("ServerColdStart", 1, telemetry.UnitCount)
		isColdStart = false
	}

	return &TwitchEmailValidatorServiceServer{sampleReporter: sampleReporter, logger: logger, Clients: clients}, nil
}

// Given an error, map it to an appropriate twirp error
func getTwirpError(err error) error {
	switch err {
	case ErrBodyDecode, storage.ErrNeedNamespace, storage.ErrNeedKey, storage.ErrNeedEmail, storage.ErrNeedOpaqueID, storage.ErrNeedVerificationCode:
		return twirp.NewError(twirp.InvalidArgument, err.Error())
	case storage.ErrValidationNotFound, storage.ErrEmailNotFound:
		return twirp.NewError(twirp.NotFound, err.Error())
	case storage.ErrValidationAlreadyExists:
		return twirp.NewError(twirp.AlreadyExists, err.Error())
	case storage.ErrBadVerificationCode:
		return twirp.NewError(twirp.Unauthenticated, err.Error())
	case storage.ErrVerificationRequestWithoutCode, storage.ErrValidationAlreadySuccessful, storage.ErrValidationAlreadyRejected, storage.ErrValidationNotRejected:
		return twirp.NewError(twirp.FailedPrecondition, err.Error())
	case storage.ErrBadVerificationCodeMaxAttempts:
		return twirp.NewError(twirp.ResourceExhausted, err.Error())
	default:
		return twirp.InternalErrorWith(err)
	}
}

func (s *TwitchEmailValidatorServiceServer) publishVerificationRequest(request *models.VerificationRequest, purpose string) {
	var recipientID string
	// For requests from users-service, the key is the recipientID
	if request.Namespace == usersServiceNamespace {
		recipientID = request.Key
	}

	log.Printf("Sending verification request for %s (opaque ID %s, recipient ID: %s, locale: %s)", purpose, request.OpaqueID, recipientID, request.Locale)
	ctx, cancel := context.WithTimeout(context.Background(), handlerTimeout)
	defer cancel()
	err := s.Clients.PushySNSPublisher.PublishVerificationRequest(ctx, request.Email, request.OpaqueID, request.Locale, purpose, recipientID, request.VerificationCode)
	if err != nil {
		logx.Error(ctx, err)
	}
}

func (s *TwitchEmailValidatorServiceServer) publishVerificationSuccess(request *models.VerificationRequest) {
	log.Printf("Sending verification success for %s", request.Namespace)
	ctx, cancel := context.WithTimeout(context.Background(), handlerTimeout)
	defer cancel()
	err := s.Clients.SuccessSNSPublisher.PublishVerificationSuccess(ctx, request.Namespace, request.Key, request.Email, request.Locale)
	if err != nil {
		logx.Error(ctx, err)
	}
}
