package server_test

import (
	"context"

	"code.justin.tv/amzn/TwitchEmailValidatorService/models"
	"code.justin.tv/amzn/TwitchEmailValidatorService/server/converter"
	"code.justin.tv/amzn/TwitchEmailValidatorService/testutils"
	evs "code.justin.tv/amzn/TwitchEmailValidatorServiceTwirp"
	"github.com/pkg/errors"
	"github.com/twitchtv/twirp"

	"github.com/stretchr/testify/mock"
)

func (s *ServerSuite) TestUnrejectSuccess() {
	namespace := testutils.RandStr(10)
	key := testutils.RandStr(10)
	email := testutils.RandStr(10)
	req := testutils.NewVerificationRequest()

	expectedVerificationRequest, err := converter.OutputVerificationRequest(req)
	s.Require().NoError(err)

	s.mockUnrejectSuccess(namespace, key, email, req)

	out, err := s.service.Unreject(context.Background(), &evs.UnrejectInput{Namespace: namespace, Key: key, Email: email})
	s.Require().NoError(err)

	s.Require().Equal(expectedVerificationRequest, out.VerificationRequest)
	s.AssertExpectations()
}

func (s *ServerSuite) TestUnrejectInvalid() {
	tests := []struct {
		mut func(input *evs.UnrejectInput)
	}{
		{func(input *evs.UnrejectInput) {
			input.Namespace = ""
		}},
		{func(input *evs.UnrejectInput) {
			input.Key = ""
		}},
		{func(input *evs.UnrejectInput) {
			input.Email = ""
		}},
	}

	for _, tt := range tests {
		input := &evs.UnrejectInput{
			Namespace: testutils.RandStr(10),
			Key:       testutils.RandStr(10),
			Email:     testutils.RandStr(10),
		}
		tt.mut(input)

		_, err := s.service.Unreject(context.Background(), input)
		s.Require().Error(err)
		s.AssertExpectations()

		twerr := err.(twirp.Error)
		s.Require().Equal(twirp.InvalidArgument, twerr.Code())
	}
}

func (s *ServerSuite) TestUnrejectError() {
	namespace := testutils.RandStr(10)
	key := testutils.RandStr(10)
	email := testutils.RandStr(10)
	s.mockUnrejectError(namespace, key, email, errors.New("o no"))

	_, err := s.service.Unreject(context.Background(), &evs.UnrejectInput{Namespace: namespace, Key: key, Email: email})
	s.Require().Error(err)
}

func (s *ServerSuite) mockUnrejectSuccess(namespace, key, email string, req *models.VerificationRequest) {
	s.Mocks.DS.On("Unreject", mock.Anything, namespace, key, email).Return(req, nil).Once()
}

func (s *ServerSuite) mockUnrejectError(namespace, key, email string, err error) {
	s.Mocks.DS.On("Unreject", mock.Anything, namespace, key, email).Return(nil, err).Once()
}
