package server

import (
	"context"

	"code.justin.tv/amzn/TwitchEmailValidatorService/server/converter"
	"code.justin.tv/amzn/TwitchEmailValidatorService/storage"
	"github.com/pkg/errors"

	evs "code.justin.tv/amzn/TwitchEmailValidatorServiceTwirp"
	"github.com/twitchtv/twirp"
)

func (s *TwitchEmailValidatorServiceServer) VerifyCode(ctx context.Context, req *evs.VerifyCodeInput) (*evs.VerifyCodeOutput, error) {
	if req.GetKey() == "" {
		return nil, twirp.RequiredArgumentError("key")
	}
	if req.GetEmail() == "" {
		return nil, twirp.RequiredArgumentError("email")
	}
	if req.GetCode() == "" {
		return nil, twirp.RequiredArgumentError("code")
	}

	namespace := req.GetNamespace()
	if namespace == "" {
		namespace = usersServiceNamespace
	}

	request, err := s.Clients.DS.VerifyCode(ctx, namespace, req.GetKey(), req.GetEmail(), req.GetCode())
	if err != nil {
		// Regenerate code if too many failed attempts or no code found
		if err == storage.ErrBadVerificationCodeMaxAttempts || err == storage.ErrVerificationRequestWithoutCode {
			regenerateRequest, rerr := s.Clients.DS.RegenerateCode(ctx, namespace, req.GetKey(), req.GetEmail())
			if rerr != nil {
				return nil, getTwirpError(rerr)
			}

			s.publishVerificationRequest(regenerateRequest, "Too many failed validation attempts")
		}

		return nil, getTwirpError(err)
	}

	verificationRequest, err := converter.OutputVerificationRequest(request)
	if err != nil {
		return nil, twirp.InternalErrorWith(errors.Wrap(err, "converting verification request"))
	}

	s.publishVerificationSuccess(request)

	return &evs.VerifyCodeOutput{VerificationRequest: verificationRequest}, nil
}
