package server_test

import (
	"context"

	"code.justin.tv/amzn/TwitchEmailValidatorService/server/converter"
	"code.justin.tv/amzn/TwitchEmailValidatorService/storage"
	"code.justin.tv/amzn/TwitchEmailValidatorService/testutils"
	evs "code.justin.tv/amzn/TwitchEmailValidatorServiceTwirp"
	"github.com/pkg/errors"
	"github.com/stretchr/testify/mock"
	"github.com/twitchtv/twirp"
)

func (s *ServerSuite) TestVerifyCodeSuccess() {
	namespace := testutils.RandStr(10)
	key := testutils.RandStr(10)
	email := testutils.RandStr(10)
	code := testutils.RandStr(10)

	req := testutils.NewVerificationRequest()

	expectedVerificationRequest, err := converter.OutputVerificationRequest(req)
	s.Require().NoError(err)

	s.Mocks.DS.On("VerifyCode", mock.Anything, namespace, key, email, code).Return(req, nil).Once()
	s.Mocks.SuccessPublisher.On(
		"PublishVerificationSuccess", mock.Anything, req.Namespace, req.Key, req.Email, req.Locale,
	).Return(nil).Once()
	out, err := s.service.VerifyCode(context.Background(), &evs.VerifyCodeInput{
		Namespace: namespace,
		Key:       key,
		Email:     email,
		Code:      code,
	})
	s.Require().NoError(err)
	s.Require().Equal(expectedVerificationRequest, out.VerificationRequest)
	s.AssertExpectations()
}

func (s *ServerSuite) TestVerifyCodeNoNamespaceSuccess() {
	key := testutils.RandStr(10)
	email := testutils.RandStr(10)
	code := testutils.RandStr(10)

	req := testutils.NewVerificationRequest()

	expectedVerificationRequest, err := converter.OutputVerificationRequest(req)
	s.Require().NoError(err)

	s.Mocks.DS.On("VerifyCode", mock.Anything, "users-service", key, email, code).Return(req, nil).Once()
	s.Mocks.SuccessPublisher.On(
		"PublishVerificationSuccess", mock.Anything, req.Namespace, req.Key, req.Email, req.Locale,
	).Return(nil).Once()
	out, err := s.service.VerifyCode(context.Background(), &evs.VerifyCodeInput{
		Key:   key,
		Email: email,
		Code:  code,
	})
	s.Require().NoError(err)
	s.Require().Equal(expectedVerificationRequest, out.VerificationRequest)
	s.AssertExpectations()
}

func (s *ServerSuite) TestVerifyCodeInvalid() {
	tests := []struct {
		mut func(input *evs.VerifyCodeInput)
	}{
		{func(input *evs.VerifyCodeInput) {
			input.Key = ""
		}},
		{func(input *evs.VerifyCodeInput) {
			input.Email = ""
		}},
		{func(input *evs.VerifyCodeInput) {
			input.Code = ""
		}},
	}

	for _, tt := range tests {
		input := &evs.VerifyCodeInput{
			Key:   testutils.RandStr(10),
			Email: testutils.RandStr(10),
			Code:  testutils.RandStr(6),
		}
		tt.mut(input)

		_, err := s.service.VerifyCode(context.Background(), input)
		s.Require().Error(err)
		s.AssertExpectations()

		twerr := err.(twirp.Error)
		s.Require().Equal(twirp.InvalidArgument, twerr.Code())
	}
}

func (s *ServerSuite) TestVerifyCodeError() {
	namespace := testutils.RandStr(10)
	key := testutils.RandStr(10)
	email := testutils.RandStr(10)
	code := testutils.RandStr(10)

	s.Mocks.DS.On("VerifyCode", mock.Anything, namespace, key, email, code).Return(nil, errors.New("o no")).Once()

	_, err := s.service.VerifyCode(context.Background(), &evs.VerifyCodeInput{
		Namespace: namespace,
		Key:       key,
		Email:     email,
		Code:      code,
	})
	s.Require().Error(err)
}

func (s *ServerSuite) TestVerifyCodeTooManyAttemptsError() {
	namespace := testutils.RandStr(10)
	key := testutils.RandStr(10)
	email := testutils.RandStr(10)
	code := testutils.RandStr(10)

	req := testutils.NewVerificationRequest()

	s.Mocks.DS.On("VerifyCode", mock.Anything, namespace, key, email, code).Return(nil, storage.ErrBadVerificationCodeMaxAttempts).Once()
	s.Mocks.DS.On("RegenerateCode", mock.Anything, namespace, key, email).Return(req, nil).Once()
	s.Mocks.PushyPublisher.On(
		"PublishVerificationRequest", mock.Anything, req.Email, req.OpaqueID, req.Locale, "Too many failed validation attempts", mock.Anything, req.VerificationCode,
	).Return(nil).Once()

	_, err := s.service.VerifyCode(context.Background(), &evs.VerifyCodeInput{
		Namespace: namespace,
		Key:       key,
		Email:     email,
		Code:      code,
	})
	s.Require().Error(err)
}

func (s *ServerSuite) TestVerifyCodeTooManyAttemptsRegenerateError() {
	namespace := testutils.RandStr(10)
	key := testutils.RandStr(10)
	email := testutils.RandStr(10)
	code := testutils.RandStr(10)

	s.Mocks.DS.On("VerifyCode", mock.Anything, namespace, key, email, code).Return(nil, storage.ErrBadVerificationCodeMaxAttempts).Once()
	s.Mocks.DS.On("RegenerateCode", mock.Anything, namespace, key, email).Return(nil, errors.New("o no")).Once()
	_, err := s.service.VerifyCode(context.Background(), &evs.VerifyCodeInput{
		Namespace: namespace,
		Key:       key,
		Email:     email,
		Code:      code,
	})
	s.Require().Error(err)
}
