package storage

import (
	"context"
	"log"
	"time"

	"code.justin.tv/amzn/TwitchEmailValidatorService/models"
	"github.com/aws/aws-sdk-go/aws/awserr"
	"github.com/aws/aws-sdk-go/service/dynamodb"
)

// CreateVerificationRequest adds a new verification request to the store
// It is an error to request verification for an email that is
// already known to be valid or rejected
func (ds *Verifier) CreateVerificationRequest(ctx context.Context, namespace, key, email, locale string, shouldIncludeCode bool) (*models.VerificationRequest, error) {
	if len(namespace) == 0 {
		return nil, ErrNeedNamespace
	}
	if len(key) == 0 {
		return nil, ErrNeedKey
	}
	if len(email) == 0 {
		return nil, ErrNeedEmail
	}

	// First let's see if we already have this request somewhere

	request, err := ds.VerificationRequest(ctx, namespace, key, email)
	if err != nil && err != ErrValidationNotFound {
		return nil, err
	}
	if request != nil {
		if request.Status == models.StatusPending {
			// This is a duplicate of an existing pending request,
			// just update the modified time and return the existing request to resend email
			err = ds.upsertRequest(ctx, request, false)
			if err != nil {
				return nil, err
			}

			return request, nil
		}
		log.Printf("Verification request already exists (namespace: %s, key: %s)", namespace, key)
		return nil, ErrValidationAlreadyExists
	}

	opaqueID, err := models.MakeOpaqueID()
	if err != nil {
		return nil, err
	}

	var verificationCode string
	if shouldIncludeCode {
		verificationCode, err = models.MakeVerificationCode()
		if err != nil {
			return nil, err
		}
	}

	request = &models.VerificationRequest{
		Namespace:            namespace,
		Key:                  key,
		Email:                email,
		Locale:               locale,
		Status:               models.StatusPending,
		Created:              time.Now(),
		Modified:             time.Now(),
		CompoundKey:          models.MakeCompoundKey(namespace, key, email),
		OpaqueID:             opaqueID,
		VerificationCode:     verificationCode,
		VerificationAttempts: 0,
	}

	err = ds.upsertRequest(ctx, request, true)
	if err != nil {
		// It's possible for two parallel CreateVerificationRequest requests to make a PutItem request. If the PutItem request
		// fails with a conditional check failed, then the other request has succeeded and the existing verification request
		// can be returned.
		if awsErr, ok := err.(awserr.Error); ok && awsErr.Code() == dynamodb.ErrCodeConditionalCheckFailedException {
			ds.reportAWSError(ctx, err, "datastore.createverificationrequest.error.putitem_conditional_check_failed")
			return ds.VerificationRequest(ctx, namespace, key, email)
		}

		ds.reportAWSError(ctx, err, "datastore.createverificationrequest.error.putitem")
		return nil, err
	}

	return request, nil
}
