package storage_test

import (
	"context"

	"code.justin.tv/amzn/TwitchEmailValidatorService/models"
	datasource "code.justin.tv/amzn/TwitchEmailValidatorService/storage"
	"code.justin.tv/amzn/TwitchEmailValidatorService/testutils"
)

func (s *VerifierSuite) TestCreateVerificationRequestSuccess() {
	namespace := testutils.RandStr(10)
	key := testutils.RandStr(10)
	email := testutils.RandEmail()

	s.MockDynamoDBGetVerificationEmpty(namespace, key, email)
	s.MockDynamoDBPutItemSuccess()

	request, err := s.Verifier.CreateVerificationRequest(context.Background(), namespace, key, email, locale, true)
	s.Require().NoError(err)

	s.Equal(namespace, request.Namespace)
	s.Equal(key, request.Key)
	s.Equal(email, request.Email)
	s.Equal(models.StatusPending, request.Status)
	s.Equal(0, request.VerificationAttempts)
	s.Equal(locale, request.Locale)
	s.Equal(models.MakeCompoundKey(namespace, key, email), request.CompoundKey)
	s.NotEmpty(request.VerificationCode)

	s.AssertExpectations()
}

func (s *VerifierSuite) TestCreateVerificationRequestExistsPendingSuccess() {
	namespace := testutils.RandStr(10)
	key := testutils.RandStr(10)
	email := testutils.RandEmail()

	s.MockDynamoDBGetVerificationSuccess(namespace, key, email, models.StatusPending)
	s.MockDynamoDBPutItemSuccess()

	_, err := s.Verifier.CreateVerificationRequest(context.Background(), namespace, key, email, locale, true)

	s.Require().NoError(err)
	s.AssertExpectations()
}

func (s *VerifierSuite) TestCreateVerificationRequestExistsRejectedFail() {
	namespace := testutils.RandStr(10)
	key := testutils.RandStr(10)
	email := testutils.RandEmail()

	s.MockDynamoDBGetVerificationSuccess(namespace, key, email, models.StatusRejected)

	_, err := s.Verifier.CreateVerificationRequest(context.Background(), namespace, key, email, locale, true)

	s.Require().Error(err)
	s.isError(datasource.ErrValidationAlreadyExists, err)
	s.AssertExpectations()
}

func (s *VerifierSuite) TestCreateVerificationRequestGetVerificationFail() {
	namespace := testutils.RandStr(10)
	key := testutils.RandStr(10)
	email := testutils.RandEmail()

	s.MockDynamoDBGetVerificationFail(namespace, key, email)

	_, err := s.Verifier.CreateVerificationRequest(context.Background(), namespace, key, email, locale, true)

	s.Require().Error(err)
	s.isError(AWSError, err)
	s.AssertExpectations()
}

func (s *VerifierSuite) TestCreateVerificationRequestPutItemFail() {
	namespace := testutils.RandStr(10)
	key := testutils.RandStr(10)
	email := testutils.RandEmail()

	s.MockDynamoDBGetVerificationEmpty(namespace, key, email)
	s.MockDynamoDBPutItemFail()

	_, err := s.Verifier.CreateVerificationRequest(context.Background(), namespace, key, email, locale, true)

	s.Require().Error(err)
	s.isError(AWSError, err)
	s.AssertExpectations()
}

func (s *VerifierSuite) TestCreateVerificationInvalidInputFail() {
	namespace := testutils.RandStr(10)
	key := testutils.RandStr(10)
	email := testutils.RandEmail()

	_, err := s.Verifier.CreateVerificationRequest(context.Background(), "", key, email, locale, true)
	s.Require().Error(err)
	s.isError(datasource.ErrNeedNamespace, err)

	_, err = s.Verifier.CreateVerificationRequest(context.Background(), namespace, "", email, locale, true)
	s.Require().Error(err)
	s.isError(datasource.ErrNeedKey, err)

	_, err = s.Verifier.CreateVerificationRequest(context.Background(), namespace, key, "", locale, true)
	s.Require().Error(err)
	s.isError(datasource.ErrNeedEmail, err)
}
