package storage

import (
	"context"
	"fmt"
	"time"

	logging "code.justin.tv/amzn/TwitchLogging"
	"code.justin.tv/chat/golibs/logx"

	"code.justin.tv/amzn/TwitchEmailValidatorService/clients"
	"code.justin.tv/amzn/TwitchEmailValidatorService/models"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/awserr"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbiface"
)

const (
	MaxVerificationCodeAttempts = 5
)

type VerifierAPI interface {
	CreateVerificationRequest(ctx context.Context, namespace, key, email, locale string, shouldIncludeCode bool) (*models.VerificationRequest, error)
	DeleteVerificationRequest(ctx context.Context, namespace, key, email string) (*models.VerificationRequest, error)
	DeleteVerificationRequestsByKey(ctx context.Context, key string) error
	ListVerificationRequestsByKey(ctx context.Context, key string, params models.ListVerificationsParams) (*models.ListVerificationsOutput, error)
	NotMe(ctx context.Context, email, opaqueID string) error
	RegenerateCode(ctx context.Context, namespace, key, email string) (*models.VerificationRequest, error)
	Reject(ctx context.Context, email string) error
	Unreject(ctx context.Context, namespace, key, email string) (*models.VerificationRequest, error)
	VerificationRequest(ctx context.Context, namespace, key, email string) (*models.VerificationRequest, error)
	VerificationRequestByOpaqueID(ctx context.Context, opaqueID string) (*models.VerificationRequest, error)
	Verify(ctx context.Context, opaqueID string) (*models.VerificationRequest, error)
	VerifyCode(ctx context.Context, namespace, key, email, code string) (*models.VerificationRequest, error)
}

// Verifier holds all the information we need to talk to Dynamo, UsersService, and gather statistics
type Verifier struct {
	DB               dynamodbiface.DynamoDBAPI
	ValidationsTable string
	User             clients.UserWrapper
	logger           logging.Logger
}

// NewDatasourceInput is the input struct to the NewDatasource factory method
type NewDatasourceInput struct {
	DB               dynamodbiface.DynamoDBAPI
	ValidationsTable string
	User             clients.UserWrapper
	Logger           logging.Logger
}

// NewDatasource constructs the Verifier data source and hooks the implementation up to a statsd gatherer.
func NewDatasource(input NewDatasourceInput) (*Verifier, error) {
	if input.DB == nil {
		return nil, ErrNeedDynamo
	}

	if len(input.ValidationsTable) == 0 {
		return nil, ErrNeedTable
	}

	if input.User == nil {
		return nil, ErrNeedUserWrapper
	}

	ds := &Verifier{
		DB:               input.DB,
		ValidationsTable: input.ValidationsTable,
		User:             input.User,
		logger:           input.Logger,
	}

	return ds, nil
}

func (ds *Verifier) reportAWSError(ctx context.Context, err error, statKey string) {
	code := "unknown"
	if awsErr, ok := err.(awserr.Error); ok {
		code = awsErr.Code()
	}
	logx.Error(ctx, err, logx.Fields{
		"key":  statKey,
		"code": code,
	})
}

func (ds *Verifier) upsertRequest(ctx context.Context, request *models.VerificationRequest, ensureUnique bool) error {
	request.Modified = time.Now()
	entry, err := dynamodbattribute.MarshalMap(request)
	if err != nil {
		errorMessage := fmt.Errorf("upsertRequest error while marshaling: %v", err)
		ds.reportAWSError(ctx, err, "datastore.createverificationrequest.error.marshal") // keeping stat around for legacy purposes
		return errorMessage
	}

	input := dynamodb.PutItemInput{
		TableName: aws.String(ds.ValidationsTable),
		Item:      entry,
	}

	if ensureUnique {
		input.ConditionExpression = aws.String("attribute_not_exists(compound_key)")
	}

	_, err = ds.DB.PutItemWithContext(ctx, &input)

	return err
}

var _ VerifierAPI = (*Verifier)(nil)
