package storage

import (
	"context"
	"fmt"

	"code.justin.tv/amzn/TwitchEmailValidatorService/models"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
)

// DeleteVerificationRequest deletes a verification request based on namespace, key, and email and returns the old state
// of the request. Used to invalidate a email when a user changes their address.
//
// Returns ErrValidationNotFound if the request did not exist or ErrValidationAlreadyRejected if the request was rejected.
func (ds *Verifier) DeleteVerificationRequest(ctx context.Context, namespace, key, email string) (*models.VerificationRequest, error) {
	output, err := ds.DB.GetItemWithContext(ctx, &dynamodb.GetItemInput{
		TableName: aws.String(ds.ValidationsTable),
		Key: map[string]*dynamodb.AttributeValue{
			"compound_key": {S: aws.String(models.MakeCompoundKey(namespace, key, email))},
		},
		ConsistentRead: aws.Bool(true),
	})

	if err != nil {
		errorMessage := fmt.Errorf("dynamodb.GetItem error: %v", err)
		ds.reportAWSError(ctx, err, "datastore.deleteverificationrequest.error.getitem")
		return nil, errorMessage
	}

	if len(output.Item) == 0 {
		return nil, ErrValidationNotFound
	}

	var oldReq models.VerificationRequest
	err = dynamodbattribute.UnmarshalMap(output.Item, &oldReq)
	if err != nil {
		ds.reportAWSError(ctx, err, "datastore.deleteverificationrequest.error.unmarshal")
		return nil, err
	}

	// Don't delete rejected requests, or else we might try to resend.
	if oldReq.Status == models.StatusRejected {
		return nil, ErrValidationAlreadyRejected
	}

	_, err = ds.DB.DeleteItemWithContext(ctx, &dynamodb.DeleteItemInput{
		TableName: aws.String(ds.ValidationsTable),
		Key: map[string]*dynamodb.AttributeValue{
			"compound_key": {S: aws.String(models.MakeCompoundKey(namespace, key, email))},
		},
	})
	if err != nil {
		errorMessage := fmt.Errorf("dynamodb.DeleteItem error: %v", err)
		ds.reportAWSError(ctx, err, "datastore.deleteverificationrequest.error.deleteitem")
		return nil, errorMessage
	}

	return &oldReq, nil
}
