package storage_test

import (
	"context"

	"code.justin.tv/amzn/TwitchEmailValidatorService/models"
	datasource "code.justin.tv/amzn/TwitchEmailValidatorService/storage"
	"code.justin.tv/amzn/TwitchEmailValidatorService/testutils"
)

func (s *VerifierSuite) TestDeleteVerificationRequestSuccess() {
	namespace := testutils.RandStr(10)
	key := testutils.RandStr(10)
	email := testutils.RandEmail()

	s.MockDynamoDBGetVerificationSuccess(namespace, key, email, models.StatusPending)
	s.MockDynamoDBDeleteItemSuccess(1)

	request, err := s.Verifier.DeleteVerificationRequest(context.Background(), namespace, key, email)

	s.Require().NoError(err)
	s.Equal(namespace, request.Namespace)
	s.Equal(key, request.Key)
	s.Equal(email, request.Email)
	s.Equal(models.StatusPending, request.Status)
	s.AssertExpectations()
}

func (s *VerifierSuite) TestDeleteVerificationRequestGetVerificationFail() {
	namespace := testutils.RandStr(10)
	key := testutils.RandStr(10)
	email := testutils.RandEmail()

	s.MockDynamoDBGetVerificationFail(namespace, key, email)

	_, err := s.Verifier.DeleteVerificationRequest(context.Background(), namespace, key, email)

	s.Require().Error(err)
	s.isError(AWSError, err)
	s.AssertExpectations()
}

func (s *VerifierSuite) TestDeleteVerificationRequestVerificationNotFoundFail() {
	namespace := testutils.RandStr(10)
	key := testutils.RandStr(10)
	email := testutils.RandEmail()

	s.MockDynamoDBGetVerificationEmpty(namespace, key, email)

	_, err := s.Verifier.DeleteVerificationRequest(context.Background(), namespace, key, email)

	s.Require().Error(err)
	s.isError(datasource.ErrValidationNotFound, err)
	s.AssertExpectations()
}

func (s *VerifierSuite) TestDeleteVerificationRequestDeleteRejectedFail() {
	namespace := testutils.RandStr(10)
	key := testutils.RandStr(10)
	email := testutils.RandEmail()

	s.MockDynamoDBGetVerificationSuccess(namespace, key, email, models.StatusRejected)
	_, err := s.Verifier.DeleteVerificationRequest(context.Background(), namespace, key, email)

	s.Require().Error(err)
	s.isError(datasource.ErrValidationAlreadyRejected, err)
	s.AssertExpectations()
}

func (s *VerifierSuite) TestDeleteVerificationAWSDeleteFail() {
	namespace := testutils.RandStr(10)
	key := testutils.RandStr(10)
	email := testutils.RandEmail()

	s.MockDynamoDBGetVerificationSuccess(namespace, key, email, models.StatusPending)
	s.MockDynamoDBDeleteItemFail()

	_, err := s.Verifier.DeleteVerificationRequest(context.Background(), namespace, key, email)

	s.Require().Error(err)
	s.isError(AWSError, err)
	s.AssertExpectations()
}
