package storage

import (
	"context"

	"code.justin.tv/amzn/TwitchEmailValidatorService/models"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/hashicorp/go-multierror"
)

// DeleteVerificationRequestsByKey deletes all verification requests associated with a key (often user ID)
func (ds *Verifier) DeleteVerificationRequestsByKey(ctx context.Context, key string) error {
	if len(key) == 0 {
		return ErrNeedKey
	}

	var continuationToken string
	var deleteErrors error
	for {
		listOut, err := ds.ListVerificationRequestsByKey(ctx, key, models.ListVerificationsParams{
			ContinuationToken: continuationToken,
			Limit:             100,
		})
		if err != nil {
			ds.reportAWSError(ctx, err, "datastore.deleteverificationrequestsbykey.error.list")
			return err
		}

		for _, req := range listOut.VerificationRequests {
			_, deleteErr := ds.DB.DeleteItemWithContext(ctx, &dynamodb.DeleteItemInput{
				TableName: &ds.ValidationsTable,
				Key: map[string]*dynamodb.AttributeValue{
					"compound_key": {S: &req.CompoundKey},
				},
			})
			if deleteErr != nil {
				deleteErrors = multierror.Append(deleteErrors, deleteErr)
			}
		}

		if continuationToken == "" {
			break
		}
	}

	if deleteErrors != nil {
		ds.reportAWSError(ctx, deleteErrors, "datastore.deleteverificationrequestsbykey.error.delete")
		return deleteErrors
	}

	return nil
}
