package storage

import (
	"context"
	"fmt"

	"code.justin.tv/amzn/TwitchEmailValidatorService/models"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
)

// VerificationRequest returns the request for the given namespace, id, email triple
// identification triple
func (ds *Verifier) VerificationRequest(ctx context.Context, namespace, key, email string) (*models.VerificationRequest, error) {
	if len(namespace) == 0 {
		return nil, ErrNeedNamespace
	}
	if len(key) == 0 {
		return nil, ErrNeedKey
	}
	if len(email) == 0 {
		return nil, ErrNeedEmail
	}

	output, err := ds.DB.GetItemWithContext(ctx, &dynamodb.GetItemInput{
		TableName: aws.String(ds.ValidationsTable),
		Key: map[string]*dynamodb.AttributeValue{
			"compound_key": {S: aws.String(models.MakeCompoundKey(namespace, key, email))},
		},
		ConsistentRead: aws.Bool(true),
	})

	if err != nil {
		errorMessage := fmt.Errorf("dynamodb.GetItem error: %v", err)
		ds.reportAWSError(ctx, err, "datastore.verificationrequest.error.getitem")
		return nil, errorMessage
	}

	if len(output.Item) == 0 {
		return nil, ErrValidationNotFound
	}

	var request models.VerificationRequest
	err = dynamodbattribute.UnmarshalMap(output.Item, &request)
	if err != nil {
		ds.reportAWSError(ctx, err, "datastore.verificationrequest.error.unmarshal")
		return nil, err
	}
	return &request, nil
}
