package storage

import (
	"context"

	"code.justin.tv/amzn/TwitchEmailValidatorService/models"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
)

// VerificationRequestByOpaqueID returns the request for the given opaque ID
func (ds *Verifier) VerificationRequestByOpaqueID(ctx context.Context, opaqueID string) (*models.VerificationRequest, error) {
	if len(opaqueID) == 0 {
		return nil, ErrNeedOpaqueID
	}

	opaqueOutput, err := ds.DB.QueryWithContext(ctx, &dynamodb.QueryInput{
		TableName:              aws.String(ds.ValidationsTable),
		KeyConditionExpression: aws.String("opaque_id = :id"),
		IndexName:              aws.String("opaque_id_index"),

		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":id": {S: aws.String(opaqueID)},
		},
	})

	if err != nil {
		ds.reportAWSError(ctx, err, "datastore.verificationrequestbyopaqueid.error.query")
		return nil, err
	}

	if len(opaqueOutput.Items) == 0 {
		return nil, ErrValidationNotFound
	}

	if len(opaqueOutput.Items) > 1 {
		return nil, ErrMultipleValidationsFound
	}

	var request models.VerificationRequest
	err = dynamodbattribute.UnmarshalMap(opaqueOutput.Items[0], &request)
	if err != nil {
		ds.reportAWSError(ctx, err, "datastore.verificationrequestbyopaqueid.error.unmarshal")
		return nil, err
	}

	return &request, nil
}
