package storage_test

import (
	"context"

	"code.justin.tv/amzn/TwitchEmailValidatorService/models"
	datasource "code.justin.tv/amzn/TwitchEmailValidatorService/storage"
	"code.justin.tv/amzn/TwitchEmailValidatorService/testutils"
)

func (s *VerifierSuite) TestVerificationRequestSuccess() {
	namespace := testutils.RandStr(10)
	key := testutils.RandStr(10)
	email := testutils.RandEmail()

	s.MockDynamoDBGetVerificationSuccess(namespace, key, email, models.StatusPending)

	request, err := s.Verifier.VerificationRequest(context.Background(), namespace, key, email)

	s.Require().NoError(err)
	s.Equal(namespace, request.Namespace)
	s.Equal(key, request.Key)
	s.Equal(email, request.Email)
	s.AssertExpectations()
}

func (s *VerifierSuite) TestVerificationRequestAWSFail() {
	namespace := testutils.RandStr(10)
	key := testutils.RandStr(10)
	email := testutils.RandEmail()

	s.MockDynamoDBGetVerificationFail(namespace, key, email)

	_, err := s.Verifier.VerificationRequest(context.Background(), namespace, key, email)

	s.Require().Error(err)
	s.isError(AWSError, err)
	s.AssertExpectations()
}

func (s *VerifierSuite) TestVerificationRequestDoesNotExist() {
	namespace := testutils.RandStr(10)
	key := testutils.RandStr(10)
	email := testutils.RandEmail()

	s.MockDynamoDBGetVerificationEmpty(namespace, key, email)

	_, err := s.Verifier.VerificationRequest(context.Background(), namespace, key, email)

	s.Require().Error(err)
	s.isError(datasource.ErrValidationNotFound, err)
	s.AssertExpectations()
}

func (s *VerifierSuite) TestVerificationRequestInvalidInputs() {
	namespace := testutils.RandStr(10)
	key := testutils.RandStr(10)
	email := testutils.RandEmail()

	_, err := s.Verifier.VerificationRequest(context.Background(), "", key, email)
	s.Require().Error(err)
	s.isError(datasource.ErrNeedNamespace, err)

	_, err = s.Verifier.VerificationRequest(context.Background(), namespace, "", email)
	s.Require().Error(err)
	s.isError(datasource.ErrNeedKey, err)

	_, err = s.Verifier.VerificationRequest(context.Background(), namespace, key, "")
	s.Require().Error(err)
	s.isError(datasource.ErrNeedEmail, err)
}
