package storage

import (
	"context"
	"encoding/json"

	"code.justin.tv/amzn/TwitchEmailValidatorService/models"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
)

const defaultListLimit uint = 10

// ListVerificationRequestsByKey lists all verification requests associated with a key (often user ID)
func (ds *Verifier) ListVerificationRequestsByKey(ctx context.Context, key string, params models.ListVerificationsParams) (*models.ListVerificationsOutput, error) {
	if len(key) == 0 {
		return nil, ErrNeedKey
	}

	limit := params.Limit
	if limit == 0 {
		limit = defaultListLimit
	}

	var startKey map[string]*dynamodb.AttributeValue
	if params.ContinuationToken != "" {
		err := json.Unmarshal([]byte(params.ContinuationToken), &startKey)
		if err != nil {
			ds.reportAWSError(ctx, err, "datastore.listverificationrequestsbykey.error.unmarshaltoken")
			return nil, err
		}
	}

	input := &dynamodb.QueryInput{
		TableName:              aws.String(ds.ValidationsTable),
		KeyConditionExpression: aws.String("#key = :key"),
		IndexName:              aws.String("key_index"),
		ExpressionAttributeNames: map[string]*string{
			"#key": aws.String("key"),
		},
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":key": {S: aws.String(key)},
		},
		Limit:             aws.Int64(int64(limit)),
		ExclusiveStartKey: startKey,
	}

	output, err := ds.DB.QueryWithContext(ctx, input)
	if err != nil {
		ds.reportAWSError(ctx, err, "datastore.listverificationrequestsbykey.error.query")
		return nil, err
	}

	requests := make([]models.VerificationRequest, 0, len(output.Items))
	for _, item := range output.Items {
		var request models.VerificationRequest
		err = dynamodbattribute.UnmarshalMap(item, &request)
		if err != nil {
			ds.reportAWSError(ctx, err, "datastore.listverificationrequestsbykey.error.unmarshal")
			return nil, err
		}

		requests = append(requests, request)
	}

	var nextContinuationToken string
	if len(output.LastEvaluatedKey) != 0 {
		b, err := json.Marshal(output.LastEvaluatedKey)
		if err != nil {
			ds.reportAWSError(ctx, err, "datastore.listverificationrequestsbykey.error.marshal")
			return nil, err
		}
		nextContinuationToken = string(b)
	}

	return &models.ListVerificationsOutput{
		VerificationRequests: requests,
		ContinuationToken:    nextContinuationToken,
	}, nil
}
