package storage_test

import (
	"context"

	"code.justin.tv/amzn/TwitchEmailValidatorService/models"
	datasource "code.justin.tv/amzn/TwitchEmailValidatorService/storage"
	"code.justin.tv/amzn/TwitchEmailValidatorService/testutils"
)

func (s *VerifierSuite) TestListVerificationRequestsByKeySuccess() {
	key := testutils.RandStr(5)
	compoundKeys := testutils.RandStringSlice(5)

	s.MockDynamoDBQueryByKeySuccess(key, compoundKeys)

	listOut, err := s.Verifier.ListVerificationRequestsByKey(context.Background(), key, models.ListVerificationsParams{})
	s.Require().NoError(err)

	s.Require().Len(listOut.VerificationRequests, len(compoundKeys))
	for i, req := range listOut.VerificationRequests {
		s.Equal(key, req.Key)
		s.Equal(compoundKeys[i], req.CompoundKey)
	}
	s.AssertExpectations()
}

func (s *VerifierSuite) TestListVerificationRequestsByKeySuccessNoRecords() {
	key := testutils.RandStr(5)

	s.MockDynamoDBQueryByKeySuccess(key, []string{})

	listOut, err := s.Verifier.ListVerificationRequestsByKey(context.Background(), key, models.ListVerificationsParams{})
	s.Require().NoError(err)

	s.Require().Len(listOut.VerificationRequests, 0)
	s.AssertExpectations()
}

func (s *VerifierSuite) TestListVerificationRequestsByKeyNoKeyFail() {
	_, err := s.Verifier.ListVerificationRequestsByKey(context.Background(), "", models.ListVerificationsParams{})
	s.Require().Error(err)
	s.isError(datasource.ErrNeedKey, err)

	s.AssertExpectations()
}

func (s *VerifierSuite) TestListVerificationRequestsByKeyBadContinuationTokenFail() {

	_, err := s.Verifier.ListVerificationRequestsByKey(context.Background(), testutils.RandStr(5), models.ListVerificationsParams{
		ContinuationToken: testutils.RandStr(10),
	})
	s.Require().Error(err)

	s.AssertExpectations()
}

func (s *VerifierSuite) TestListVerificationRequestsByKeyQueryErrorFail() {
	key := testutils.RandStr(5)

	s.MockDynamoDBQueryByKeyFail(key)

	_, err := s.Verifier.ListVerificationRequestsByKey(context.Background(), key, models.ListVerificationsParams{})
	s.Require().Error(err)
	s.isError(AWSError, err)

	s.AssertExpectations()
}
