package storage_test

import (
	"context"

	datasource "code.justin.tv/amzn/TwitchEmailValidatorService/storage"
	"code.justin.tv/amzn/TwitchEmailValidatorService/testutils"
)

func (s *VerifierSuite) TestNotMeSuccess() {
	email := testutils.RandEmail()
	opaqueID := testutils.RandStr(10)

	s.MockDynamoDBQueryOpaqueVerificationSuccess(opaqueID, email)
	s.MockEventualHardDeleteUserSuccess()

	err := s.Verifier.NotMe(context.Background(), email, opaqueID)

	s.Require().NoError(err)
	s.AssertExpectations()
}

func (s *VerifierSuite) TestNotMeGetVerificationByOpaqueIDFail() {
	email := testutils.RandEmail()
	opaqueID := testutils.RandStr(10)

	s.MockDynamoDBQueryOpaqueVerificationFail(opaqueID)

	err := s.Verifier.NotMe(context.Background(), email, opaqueID)

	s.Require().Error(err)
	s.isError(AWSError, err)
	s.AssertExpectations()
}

func (s *VerifierSuite) TestNotMeGetWrongEmailFail() {
	email := testutils.RandEmail()
	otherEmail := testutils.RandEmail()
	opaqueID := testutils.RandStr(10)

	s.MockDynamoDBQueryOpaqueVerificationSuccess(opaqueID, otherEmail)

	err := s.Verifier.NotMe(context.Background(), email, opaqueID)

	s.Require().Error(err)
	s.isError(datasource.ErrValidationNotFound, err)
	s.AssertExpectations()
}

func (s *VerifierSuite) TestNotMeDeleteUserFail() {
	email := testutils.RandEmail()
	opaqueID := testutils.RandStr(10)

	s.MockDynamoDBQueryOpaqueVerificationSuccess(opaqueID, email)
	s.MockEventualHardDeleteUserFail()

	err := s.Verifier.NotMe(context.Background(), email, opaqueID)

	s.Require().Error(err)
	s.isError(UserDeleteFailure, err)
	s.AssertExpectations()
}

func (s *VerifierSuite) TestNotMeInvalidInputFail() {
	email := testutils.RandEmail()
	opaqueID := testutils.RandStr(10)

	err := s.Verifier.NotMe(context.Background(), "", opaqueID)
	s.Require().Error(err)
	s.isError(datasource.ErrNeedEmail, err)

	err = s.Verifier.NotMe(context.Background(), email, "")
	s.Require().Error(err)
	s.isError(datasource.ErrNeedOpaqueID, err)
}
