package storage

import (
	"context"
	"time"

	"code.justin.tv/amzn/TwitchEmailValidatorService/models"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/awserr"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
)

// RegenerateCode generates a new 6-digit verification code for a given
// user/email combination.  Can be requested manually or triggered when the
// user exceeds the maximum number of failed verification attempts.
func (ds *Verifier) RegenerateCode(ctx context.Context, namespace, key, email string) (*models.VerificationRequest, error) {
	if len(namespace) == 0 {
		return nil, ErrNeedNamespace
	}
	if len(key) == 0 {
		return nil, ErrNeedKey
	}
	if len(email) == 0 {
		return nil, ErrNeedEmail
	}

	existingReq, err := ds.VerificationRequest(ctx, namespace, key, email)
	if err != nil {
		return nil, err
	}

	// Block sending an extra email if already verified or rejected
	if existingReq.Status == models.StatusVerified {
		return nil, ErrValidationAlreadySuccessful
	} else if existingReq.Status == models.StatusRejected {
		return nil, ErrValidationAlreadyRejected
	}

	verificationCode, err := models.MakeVerificationCode()
	if err != nil {
		return nil, err
	}

	output, err := ds.DB.UpdateItemWithContext(ctx, &dynamodb.UpdateItemInput{
		TableName: aws.String(ds.ValidationsTable),
		Key: map[string]*dynamodb.AttributeValue{
			"compound_key": {
				S: aws.String(models.MakeCompoundKey(namespace, key, email)),
			},
		},
		UpdateExpression:    aws.String("SET #va = :verification_attempts, #vc = :verification_code, #m = :modified"),
		ConditionExpression: aws.String("#ck = :compound_key"),
		ExpressionAttributeNames: map[string]*string{
			"#ck": aws.String("compound_key"),
			"#va": aws.String("verification_attempts"),
			"#vc": aws.String("verification_code"),
			"#m":  aws.String("modified"),
		},
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":compound_key":          {S: aws.String(models.MakeCompoundKey(namespace, key, email))},
			":verification_attempts": {N: aws.String("0")},
			":verification_code":     {S: aws.String(verificationCode)},
			":modified":              {S: aws.String(time.Now().Format(time.RFC3339Nano))},
		},
		ReturnValues: aws.String(dynamodb.ReturnValueAllNew),
	})
	if err != nil {
		// Conditional check failure is expected and means the item was not found
		awsErr, ok := err.(awserr.Error)
		if ok && awsErr.Code() == dynamodb.ErrCodeConditionalCheckFailedException {
			return nil, ErrValidationNotFound
		}

		ds.reportAWSError(ctx, err, "datastore.regeneratecode.updateitem")
		return nil, err
	}

	var request models.VerificationRequest
	err = dynamodbattribute.UnmarshalMap(output.Attributes, &request)
	if err != nil {
		ds.reportAWSError(ctx, err, "datastore.verifycode.error.unmarshal")
		return nil, err
	}

	return &request, nil
}
