package storage_test

import (
	"context"

	"code.justin.tv/amzn/TwitchEmailValidatorService/models"
	datasource "code.justin.tv/amzn/TwitchEmailValidatorService/storage"
	"code.justin.tv/amzn/TwitchEmailValidatorService/testutils"
)

func (s *VerifierSuite) TestRegenerateCodeSuccess() {
	namespace := testutils.RandStr(10)
	key := testutils.RandStr(10)
	email := testutils.RandEmail()

	s.MockDynamoDBGetVerificationSuccess(namespace, key, email, models.StatusPending)
	s.MockDynamoDBUpdateItemSuccess(namespace, key, email)

	_, err := s.Verifier.RegenerateCode(context.Background(), namespace, key, email)
	s.Require().NoError(err)
	s.AssertExpectations()
}

func (s *VerifierSuite) TestRegenerateCodeGetVerificationFail() {
	namespace := testutils.RandStr(10)
	key := testutils.RandStr(10)
	email := testutils.RandEmail()

	s.MockDynamoDBGetVerificationFail(namespace, key, email)

	_, err := s.Verifier.RegenerateCode(context.Background(), namespace, key, email)
	s.Require().Error(err)
	s.isError(AWSError, err)
	s.AssertExpectations()
}

func (s *VerifierSuite) TestRegenerateCodeAlreadyVerifiedFail() {
	namespace := testutils.RandStr(10)
	key := testutils.RandStr(10)
	email := testutils.RandEmail()

	s.MockDynamoDBGetVerificationSuccess(namespace, key, email, models.StatusVerified)

	_, err := s.Verifier.RegenerateCode(context.Background(), namespace, key, email)
	s.Require().Error(err)
	s.isError(datasource.ErrValidationAlreadySuccessful, err)
	s.AssertExpectations()
}

func (s *VerifierSuite) TestRegenerateCodeAlreadyRejectedFail() {
	namespace := testutils.RandStr(10)
	key := testutils.RandStr(10)
	email := testutils.RandEmail()

	s.MockDynamoDBGetVerificationSuccess(namespace, key, email, models.StatusRejected)

	_, err := s.Verifier.RegenerateCode(context.Background(), namespace, key, email)
	s.Require().Error(err)
	s.isError(datasource.ErrValidationAlreadyRejected, err)
	s.AssertExpectations()
}

func (s *VerifierSuite) TestRegenerateCodeUpdateFail() {
	namespace := testutils.RandStr(10)
	key := testutils.RandStr(10)
	email := testutils.RandEmail()

	s.MockDynamoDBGetVerificationSuccess(namespace, key, email, models.StatusPending)
	s.MockDynamoDBUpdateItemUpdateFail(namespace, key, email)

	_, err := s.Verifier.RegenerateCode(context.Background(), namespace, key, email)
	s.Require().Error(err)
	s.isError(AWSError, err)
	s.AssertExpectations()
}

func (s *VerifierSuite) TestRegenerateCodeInvalidInputsFail() {
	namespace := testutils.RandStr(10)
	key := testutils.RandStr(10)
	email := testutils.RandEmail()

	_, err := s.Verifier.RegenerateCode(context.Background(), "", key, email)
	s.Require().Error(err)
	s.isError(datasource.ErrNeedNamespace, err)

	_, err = s.Verifier.RegenerateCode(context.Background(), namespace, "", email)
	s.Require().Error(err)
	s.isError(datasource.ErrNeedKey, err)

	_, err = s.Verifier.RegenerateCode(context.Background(), namespace, key, "")
	s.Require().Error(err)
	s.isError(datasource.ErrNeedEmail, err)
}
