package storage

import (
	"context"

	"code.justin.tv/amzn/TwitchEmailValidatorService/models"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
)

// Reject sets a request to "Rejected" given the user's email address only.
// This is done in response to a bounce, and will mark ALL uses of this email
// address as rejected, even outside any namespace / ID
func (ds *Verifier) Reject(ctx context.Context, email string) error {
	if len(email) == 0 {
		return ErrNeedEmail
	}

	emailOutput, err := ds.DB.QueryWithContext(ctx, &dynamodb.QueryInput{
		TableName:              aws.String(ds.ValidationsTable),
		KeyConditionExpression: aws.String("email = :email"),
		IndexName:              aws.String("email_index"),
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":email": {S: aws.String(email)},
		},
	})

	if err != nil {
		ds.reportAWSError(ctx, err, "datastore.reject.error.query")
		return err
	}

	if len(emailOutput.Items) == 0 {
		return ErrEmailNotFound
	}

	for _, item := range emailOutput.Items {
		var request models.VerificationRequest
		err = dynamodbattribute.UnmarshalMap(item, &request)
		if err != nil {
			ds.reportAWSError(ctx, err, "datastore.reject.error.unmarshal")
			return err
		}

		request.Status = models.StatusRejected
		err = ds.upsertRequest(ctx, &request, false)
		if err != nil {
			ds.reportAWSError(ctx, err, "datastore.reject.updateitem")
			return err
		}
	}

	return nil
}
