package storage_test

import (
	"errors"
	"strconv"
	"strings"
	"testing"

	"code.justin.tv/amzn/TwitchEmailValidatorService/mocks"
	"code.justin.tv/amzn/TwitchEmailValidatorService/models"
	datasource "code.justin.tv/amzn/TwitchEmailValidatorService/storage"
	"code.justin.tv/amzn/TwitchEmailValidatorService/testutils"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"
)

const locale = "en"

var AWSError = errors.New("aws failure")
var UserDeleteFailure = errors.New("user delete failure")

type VerifierSuite struct {
	suite.Suite
	*datasource.Verifier

	mockDynamoDB *mocks.DynamoDBAPI
	mockUser     *mocks.UserWrapper
}

func (s *VerifierSuite) SetupTest() {
	s.mockDynamoDB = &mocks.DynamoDBAPI{}
	s.mockUser = &mocks.UserWrapper{}

	newDatasourceInput := datasource.NewDatasourceInput{
		DB:               s.mockDynamoDB,
		ValidationsTable: "validation_table",
		User:             s.mockUser,
		Logger:           testutils.LoggerStub{},
	}
	repo, err := datasource.NewDatasource(newDatasourceInput)
	s.Require().NoError(err)

	s.Verifier = repo
}

func (s *VerifierSuite) AssertExpectations() {
	s.mockDynamoDB.AssertExpectations(s.T())
	s.mockUser.AssertExpectations(s.T())
}

func TestVerifierSuite(t *testing.T) {
	suite.Run(t, new(VerifierSuite))
}

// helpers

func (s *VerifierSuite) MockDynamoDBGetVerificationSuccess(namespace string, key string, email string, status models.VerificationStatus) {
	compoundKey := models.MakeCompoundKey(namespace, key, email)
	matcher := mock.MatchedBy(func(input *dynamodb.GetItemInput) bool {
		return *input.Key["compound_key"].S == compoundKey
	})
	s.mockDynamoDB.On("GetItemWithContext", mock.Anything, matcher).Return(&dynamodb.GetItemOutput{
		Item: map[string]*dynamodb.AttributeValue{
			"namespace": {S: &namespace},
			"key":       {S: &key},
			"email":     {S: &email},
			"status":    {N: aws.String(strconv.Itoa(int(status)))},
		},
	}, nil).Once()
}

func (s *VerifierSuite) MockDynamoDBGetVerificationFail(namespace string, key string, email string) {
	compoundKey := models.MakeCompoundKey(namespace, key, email)
	matcher := mock.MatchedBy(func(input *dynamodb.GetItemInput) bool {
		return *input.Key["compound_key"].S == compoundKey
	})
	s.mockDynamoDB.On("GetItemWithContext", mock.Anything, matcher).Return(nil, AWSError).Once()
}

func (s *VerifierSuite) MockDynamoDBGetVerificationEmpty(namespace string, key string, email string) {
	compoundKey := models.MakeCompoundKey(namespace, key, email)
	matcher := mock.MatchedBy(func(input *dynamodb.GetItemInput) bool {
		return *input.Key["compound_key"].S == compoundKey
	})
	s.mockDynamoDB.On("GetItemWithContext", mock.Anything, matcher).Return(&dynamodb.GetItemOutput{}, nil).Once()
}

func (s *VerifierSuite) MockDynamoDBQueryOpaqueVerificationSuccess(id string, email string) {
	matcher := mock.MatchedBy(func(input *dynamodb.QueryInput) bool {
		return *input.ExpressionAttributeValues[":id"].S == id
	})
	key := testutils.RandStr(15)
	s.mockDynamoDB.On("QueryWithContext", mock.Anything, matcher).Return(&dynamodb.QueryOutput{
		Items: []map[string]*dynamodb.AttributeValue{{
			"opaque_id": {S: &id},
			"email":     {S: &email},
			"key":       {S: &key},
		}},
	}, nil).Once()
}

func (s *VerifierSuite) MockDynamoDBQueryOpaqueVerificationFail(id string) {
	matcher := mock.MatchedBy(func(input *dynamodb.QueryInput) bool {
		return *input.ExpressionAttributeValues[":id"].S == id
	})
	s.mockDynamoDB.On("QueryWithContext", mock.Anything, matcher).Return(nil, AWSError).Once()
}

func (s *VerifierSuite) MockDynamoDBQueryOpaqueVerificationEmpty(id string) {
	matcher := mock.MatchedBy(func(input *dynamodb.QueryInput) bool {
		return *input.ExpressionAttributeValues[":id"].S == id
	})
	s.mockDynamoDB.On("QueryWithContext", mock.Anything, matcher).Return(&dynamodb.QueryOutput{
		Items: []map[string]*dynamodb.AttributeValue{},
	}, nil).Once()
}

func (s *VerifierSuite) MockDynamoDBQueryOpaqueVerificationMultiple(id string) {
	matcher := mock.MatchedBy(func(input *dynamodb.QueryInput) bool {
		return *input.ExpressionAttributeValues[":id"].S == id
	})
	s.mockDynamoDB.On("QueryWithContext", mock.Anything, matcher).Return(&dynamodb.QueryOutput{
		Items: []map[string]*dynamodb.AttributeValue{{
			"opaque_id": {S: &id},
		}, {
			"opaque_id": {S: &id},
		}},
	}, nil).Once()
}

func (s *VerifierSuite) MockDynamoDBQueryByEmailSuccess(email string) {
	matcher := mock.MatchedBy(func(input *dynamodb.QueryInput) bool {
		return *input.ExpressionAttributeValues[":email"].S == email
	})
	s.mockDynamoDB.On("QueryWithContext", mock.Anything, matcher).Return(&dynamodb.QueryOutput{
		Items: []map[string]*dynamodb.AttributeValue{{
			"email": {S: &email},
		}},
	}, nil).Once()
}

func (s *VerifierSuite) MockDynamoDBQueryByEmailEmpty(email string) {
	matcher := mock.MatchedBy(func(input *dynamodb.QueryInput) bool {
		return *input.ExpressionAttributeValues[":email"].S == email
	})
	s.mockDynamoDB.On("QueryWithContext", mock.Anything, matcher).Return(&dynamodb.QueryOutput{
		Items: []map[string]*dynamodb.AttributeValue{},
	}, nil).Once()
}

func (s *VerifierSuite) MockDynamoDBQueryByEmailFail(email string) {
	matcher := mock.MatchedBy(func(input *dynamodb.QueryInput) bool {
		return *input.ExpressionAttributeValues[":email"].S == email
	})
	s.mockDynamoDB.On("QueryWithContext", mock.Anything, matcher).Return(nil, AWSError).Once()
}

func (s *VerifierSuite) MockDynamoDBQueryByKeySuccess(key string, compoundKeys []string) {
	matcher := mock.MatchedBy(func(input *dynamodb.QueryInput) bool {
		return *input.ExpressionAttributeValues[":key"].S == key
	})

	items := make([]map[string]*dynamodb.AttributeValue, 0, len(compoundKeys))
	for i := 0; i < len(compoundKeys); i++ {
		compoundKey := compoundKeys[i]
		items = append(items, map[string]*dynamodb.AttributeValue{
			"compound_key": {S: &compoundKey},
			"key":          {S: &key},
		})
	}

	s.mockDynamoDB.On("QueryWithContext", mock.Anything, matcher).Return(&dynamodb.QueryOutput{Items: items}, nil).Once()
}

func (s *VerifierSuite) MockDynamoDBQueryByKeyFail(key string) {
	matcher := mock.MatchedBy(func(input *dynamodb.QueryInput) bool {
		return *input.ExpressionAttributeValues[":key"].S == key
	})
	s.mockDynamoDB.On("QueryWithContext", mock.Anything, matcher).Return(nil, AWSError).Once()
}

func (s *VerifierSuite) MockDynamoDBUpdateItemSuccess(namespace string, key string, email string) {
	compoundKey := models.MakeCompoundKey(namespace, key, email)
	matcher := mock.MatchedBy(func(input *dynamodb.UpdateItemInput) bool {
		return *input.Key["compound_key"].S == compoundKey
	})
	s.mockDynamoDB.On("UpdateItemWithContext", mock.Anything, matcher).Return(&dynamodb.UpdateItemOutput{
		Attributes: map[string]*dynamodb.AttributeValue{},
	}, nil).Once()
}

func (s *VerifierSuite) MockDynamoDBUpdateItemUpdateFail(namespace string, key string, email string) {
	compoundKey := models.MakeCompoundKey(namespace, key, email)
	matcher := mock.MatchedBy(func(input *dynamodb.UpdateItemInput) bool {
		return *input.Key["compound_key"].S == compoundKey
	})
	s.mockDynamoDB.On("UpdateItemWithContext", mock.Anything, matcher).Return(nil, AWSError).Once()
}

func (s *VerifierSuite) MockDynamoDBGetVerificationForVerifyCodeSuccess(namespace string, key string, email string, code string, attempts int) {
	compoundKey := models.MakeCompoundKey(namespace, key, email)
	matcher := mock.MatchedBy(func(input *dynamodb.GetItemInput) bool {
		return *input.Key["compound_key"].S == compoundKey
	})
	s.mockDynamoDB.On("GetItemWithContext", mock.Anything, matcher).Return(&dynamodb.GetItemOutput{
		Item: map[string]*dynamodb.AttributeValue{
			"namespace":             {S: &namespace},
			"key":                   {S: &key},
			"email":                 {S: &email},
			"verification_code":     {S: &code},
			"verification_attempts": {N: aws.String(strconv.Itoa(attempts))},
		},
	}, nil).Once()
}

func (s *VerifierSuite) MockDynamoDBPutItemSuccess() {
	s.mockDynamoDB.On("PutItemWithContext", mock.Anything, mock.Anything).Return(nil, nil).Once()
}

func (s *VerifierSuite) MockDynamoDBPutItemFail() {
	s.mockDynamoDB.On("PutItemWithContext", mock.Anything, mock.Anything).Return(nil, AWSError).Once()
}

func (s *VerifierSuite) MockDynamoDBDeleteItemSuccess(times int) {
	s.mockDynamoDB.On("DeleteItemWithContext", mock.Anything, mock.Anything).Return(nil, nil).Times(times)
}

func (s *VerifierSuite) MockDynamoDBDeleteItemFail() {
	s.mockDynamoDB.On("DeleteItemWithContext", mock.Anything, mock.Anything).Return(nil, AWSError).Once()
}

func (s *VerifierSuite) MockEventualHardDeleteUserSuccess() {
	s.mockUser.On("EventualHardDeleteUser", mock.Anything, mock.Anything).Return(nil).Once()
}

func (s *VerifierSuite) MockEventualHardDeleteUserFail() {
	s.mockUser.On("EventualHardDeleteUser", mock.Anything, mock.Anything).Return(UserDeleteFailure).Once()
}

func (s *VerifierSuite) isError(expected error, actual error) {
	s.True(strings.Contains(actual.Error(), expected.Error()))
}
