package storage

import (
	"context"
	"crypto/subtle"
	"fmt"
	"time"

	"code.justin.tv/amzn/TwitchEmailValidatorService/models"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/awserr"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
)

// VerifyCode sets a request to "Verified" given correct user information and
// 6-digit mobile code.
//
// In the case of an invalid code, it increments the verification attempt
// count, returning ErrBadVerificationCodeMaxAttempts when it exceeds
// MaxVerificationCodeAttempts. In case no code exists to compare against, we
// return ErrVerificationRequestWithoutCode.
func (ds *Verifier) VerifyCode(ctx context.Context, namespace, key, email, code string) (*models.VerificationRequest, error) {
	if len(namespace) == 0 {
		return nil, ErrNeedNamespace
	}
	if len(key) == 0 {
		return nil, ErrNeedKey
	}
	if len(email) == 0 {
		return nil, ErrNeedEmail
	}
	if len(code) == 0 {
		return nil, ErrNeedVerificationCode
	}

	output, err := ds.DB.GetItemWithContext(ctx, &dynamodb.GetItemInput{
		TableName: aws.String(ds.ValidationsTable),
		Key: map[string]*dynamodb.AttributeValue{
			"compound_key": {S: aws.String(models.MakeCompoundKey(namespace, key, email))},
		},
		ConsistentRead: aws.Bool(true),
	})

	if err != nil {
		errorMessage := fmt.Errorf("dynamodb.GetItem error: %v", err)
		ds.reportAWSError(ctx, err, "datastore.verifycode.error.getitem")
		return nil, errorMessage
	}

	if len(output.Item) == 0 {
		return nil, ErrValidationNotFound
	}

	var request models.VerificationRequest
	err = dynamodbattribute.UnmarshalMap(output.Item, &request)
	if err != nil {
		ds.reportAWSError(ctx, err, "datastore.verifycode.error.unmarshal")
		return nil, err
	}

	// If a user tries to verify via code but the verification request was generated before they were in the experiment
	// group, we get in this state.
	if len(request.VerificationCode) == 0 {
		return nil, ErrVerificationRequestWithoutCode
	}

	if subtle.ConstantTimeCompare([]byte(code), []byte(request.VerificationCode)) != 1 {
		attempts := request.VerificationAttempts + 1

		_, uerr := ds.DB.UpdateItemWithContext(ctx, &dynamodb.UpdateItemInput{
			TableName: aws.String(ds.ValidationsTable),
			Key: map[string]*dynamodb.AttributeValue{
				"compound_key": {
					S: aws.String(models.MakeCompoundKey(namespace, key, email)),
				},
			},
			UpdateExpression:    aws.String("SET #va = :verification_attempts, #m = :modified"),
			ConditionExpression: aws.String("#va = :old_attempts and #vc = :old_code"),
			ExpressionAttributeNames: map[string]*string{
				"#va": aws.String("verification_attempts"),
				"#vc": aws.String("verification_code"),
				"#m":  aws.String("modified"),
			},
			ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
				":verification_attempts": {N: aws.String(fmt.Sprintf("%d", attempts))},
				":old_attempts":          {N: aws.String(fmt.Sprintf("%d", attempts-1))},
				":old_code":              {S: aws.String(request.VerificationCode)},
				":modified":              {S: aws.String(time.Now().Format(time.RFC3339Nano))},
			},
		})

		if uerr != nil {
			// Conditional check failing means race condition, so just return failure without incrementing attempts
			awsErr, ok := uerr.(awserr.Error)
			if ok && awsErr.Code() == dynamodb.ErrCodeConditionalCheckFailedException {
				return nil, ErrBadVerificationCode
			}

			ds.reportAWSError(ctx, uerr, "datastore.verifycode.updateitem")
			return nil, uerr
		}

		if attempts < MaxVerificationCodeAttempts {
			return nil, ErrBadVerificationCode
		} else {
			return nil, ErrBadVerificationCodeMaxAttempts
		}
	}
	request.Status = models.StatusVerified
	err = ds.upsertRequest(ctx, &request, false)
	if err != nil {
		ds.reportAWSError(ctx, err, "datastore.verifycode.updateitem")
		return nil, err
	}

	return &request, nil
}
