package storage_test

import (
	"context"

	"code.justin.tv/amzn/TwitchEmailValidatorService/models"
	datasource "code.justin.tv/amzn/TwitchEmailValidatorService/storage"
	"code.justin.tv/amzn/TwitchEmailValidatorService/testutils"
)

func (s *VerifierSuite) TestVerifyCodeSuccess() {
	namespace := testutils.RandStr(10)
	key := testutils.RandStr(10)
	email := testutils.RandEmail()
	code := testutils.RandStr(6)

	s.MockDynamoDBGetVerificationForVerifyCodeSuccess(namespace, key, email, code, 0)
	s.MockDynamoDBPutItemSuccess()

	request, err := s.Verifier.VerifyCode(context.Background(), namespace, key, email, code)
	s.Require().NoError(err)
	s.Equal(models.StatusVerified, request.Status)
	s.AssertExpectations()
}

func (s *VerifierSuite) TestVerifyCodeGetVerificationFail() {
	namespace := testutils.RandStr(10)
	key := testutils.RandStr(10)
	email := testutils.RandEmail()
	code := testutils.RandStr(6)

	s.MockDynamoDBGetVerificationFail(namespace, key, email)

	_, err := s.Verifier.VerifyCode(context.Background(), namespace, key, email, code)
	s.Require().Error(err)
	s.isError(AWSError, err)
	s.AssertExpectations()
}

func (s *VerifierSuite) TestVerifyCodeGetVerificationNotFoundFail() {
	namespace := testutils.RandStr(10)
	key := testutils.RandStr(10)
	email := testutils.RandEmail()
	code := testutils.RandStr(6)

	s.MockDynamoDBGetVerificationEmpty(namespace, key, email)

	_, err := s.Verifier.VerifyCode(context.Background(), namespace, key, email, code)
	s.Require().Error(err)
	s.isError(datasource.ErrValidationNotFound, err)
	s.AssertExpectations()
}

func (s *VerifierSuite) TestVerifyCodeGetVerificationMissingCodeFail() {
	namespace := testutils.RandStr(10)
	key := testutils.RandStr(10)
	email := testutils.RandEmail()
	code := testutils.RandStr(6)

	s.MockDynamoDBGetVerificationForVerifyCodeSuccess(namespace, key, email, "", 0)

	_, err := s.Verifier.VerifyCode(context.Background(), namespace, key, email, code)
	s.Require().Error(err)
	s.isError(datasource.ErrVerificationRequestWithoutCode, err)
	s.AssertExpectations()
}

func (s *VerifierSuite) TestVerifyCodeGetVerificationIncorrectCodeFail() {
	namespace := testutils.RandStr(10)
	key := testutils.RandStr(10)
	email := testutils.RandEmail()
	incorrectCode := testutils.RandStr(6)
	correctCode := testutils.RandStr(6)

	s.MockDynamoDBGetVerificationForVerifyCodeSuccess(namespace, key, email, correctCode, 0)
	s.MockDynamoDBUpdateItemSuccess(namespace, key, email)

	_, err := s.Verifier.VerifyCode(context.Background(), namespace, key, email, incorrectCode)
	s.Require().Error(err)
	s.isError(datasource.ErrBadVerificationCode, err)
	s.AssertExpectations()
}

func (s *VerifierSuite) TestVerifyCodeGetVerificationTooManyAttemptsFail() {
	namespace := testutils.RandStr(10)
	key := testutils.RandStr(10)
	email := testutils.RandEmail()
	incorrectCode := testutils.RandStr(6)
	correctCode := testutils.RandStr(6)

	s.MockDynamoDBGetVerificationForVerifyCodeSuccess(namespace, key, email, correctCode, datasource.MaxVerificationCodeAttempts-1)
	s.MockDynamoDBUpdateItemSuccess(namespace, key, email)

	_, err := s.Verifier.VerifyCode(context.Background(), namespace, key, email, incorrectCode)
	s.Require().Error(err)
	s.isError(datasource.ErrBadVerificationCodeMaxAttempts, err)
	s.AssertExpectations()
}

func (s *VerifierSuite) TestVerifyCodeGetVerificationUpdateItemFail() {
	namespace := testutils.RandStr(10)
	key := testutils.RandStr(10)
	email := testutils.RandEmail()
	incorrectCode := testutils.RandStr(6)
	correctCode := testutils.RandStr(6)

	s.MockDynamoDBGetVerificationForVerifyCodeSuccess(namespace, key, email, correctCode, datasource.MaxVerificationCodeAttempts-1)
	s.MockDynamoDBUpdateItemUpdateFail(namespace, key, email)

	_, err := s.Verifier.VerifyCode(context.Background(), namespace, key, email, incorrectCode)
	s.Require().Error(err)
	s.isError(AWSError, err)
	s.AssertExpectations()
}

func (s *VerifierSuite) TestVerifyCodePutItemFail() {
	namespace := testutils.RandStr(10)
	key := testutils.RandStr(10)
	email := testutils.RandEmail()
	code := testutils.RandStr(6)

	s.MockDynamoDBGetVerificationForVerifyCodeSuccess(namespace, key, email, code, 0)
	s.MockDynamoDBPutItemFail()

	_, err := s.Verifier.VerifyCode(context.Background(), namespace, key, email, code)
	s.Require().Error(err)
	s.isError(AWSError, err)
	s.AssertExpectations()
}

func (s *VerifierSuite) TestVerifyCodeInvalidInputsFail() {
	namespace := testutils.RandStr(10)
	key := testutils.RandStr(10)
	email := testutils.RandEmail()
	code := testutils.RandStr(6)

	_, err := s.Verifier.VerifyCode(context.Background(), "", key, email, code)
	s.Require().Error(err)
	s.isError(datasource.ErrNeedNamespace, err)

	_, err = s.Verifier.VerifyCode(context.Background(), namespace, "", email, code)
	s.Require().Error(err)
	s.isError(datasource.ErrNeedKey, err)

	_, err = s.Verifier.VerifyCode(context.Background(), namespace, key, "", code)
	s.Require().Error(err)
	s.isError(datasource.ErrNeedEmail, err)

	_, err = s.Verifier.VerifyCode(context.Background(), namespace, key, email, "")
	s.Require().Error(err)
	s.isError(datasource.ErrNeedVerificationCode, err)
}
