// This file contains utility functions for manipulating and otherwise
// dealing with IP addresses, because Go's primitives for working with IP addresses are
// eccentric.
package TwitchEntryPopConsumerTwirp

import (
	"errors"
	"net"
)

var (
	ErrMissingAddress       = errors.New("protocol error: missing address")
	ErrInvalidAddressFamily = errors.New("protocol error: bad or unknown address family")
	ErrBadAddressForFamily  = errors.New("protocol error: invalid address for supplied address family")
)

func (prefix *IPPrefix) ToNetIPNet() (net.IPNet, error) {
	if prefix == nil {
		return net.IPNet{}, ErrMissingAddress
	}

	// How many bits are in addresses in our desired address family?
	var addrBits int
	if prefix.AddressFamily == IPVersion_IPv4 {
		addrBits = 32
	} else if prefix.AddressFamily == IPVersion_IPv6 {
		addrBits = 128
	} else {
		return net.IPNet{}, ErrInvalidAddressFamily
	}

	n := net.IPNet{
		IP:   net.IP(prefix.Address),
		Mask: net.CIDRMask(int(prefix.PrefixLen), addrBits),
	}

	// Validate that what we have makes some kind of sense
	if prefix.AddressFamily == IPVersion_IPv4 {
		pp := n.IP.To4()
		if pp == nil {
			return net.IPNet{}, ErrBadAddressForFamily
		}
	} else if prefix.AddressFamily == IPVersion_IPv6 {
		pp := n.IP.To16()
		if pp == nil {
			return net.IPNet{}, ErrBadAddressForFamily
		}
	}

	return n, nil
}

func PrefixFromNetIPNet(addressFamily IPVersion, p net.IPNet) *IPPrefix {
	prefixLen, _ := p.Mask.Size()
	return &IPPrefix{
		Address:       p.IP,
		PrefixLen:     uint32(prefixLen),
		AddressFamily: addressFamily,
	}
}
