package TwitchEntryPopConsumerTwirp

import (
	"testing"
	"net"
	"reflect"
	"math/rand"
)

func TestTwirpPrefixToNetIPNet(t *testing.T) {
	// First, we define our test case type and how to run one.  
	type testcase struct {
		name           string
		protoPrefix    *IPPrefix
		expectedReturn net.IPNet
		expectedErr    error     
	}	

	runTest := func (tst testcase) {
		ret, err := tst.protoPrefix.ToNetIPNet()
		
		// Did we return the right error?
		if !reflect.DeepEqual(err, tst.expectedErr) {
			t.Errorf("%s: unexpected error value; expected %v, got %v", tst.name, 
				tst.expectedErr, err)
		}
		
		// Did we return the right value?
		if !reflect.DeepEqual(ret, tst.expectedReturn) {
			t.Errorf("%s: unexpected return value; expected %v, got %+v", tst.name, 
				tst.expectedReturn, ret)
		}
	}
	
	// netonly is a utility function so we can use net.ParseCIDR in our test case 
	// definitions while ignoring its extra returns/
	netonly := func(_ net.IP, n *net.IPNet, _ error) net.IPNet {
		return *n
	}
	
	// Test cases!
	
	tests := []testcase{
	
		// First test: a nil prefix should return an error.
		{
			"nil prefix should return error",
			nil,
			net.IPNet{},
			ErrMissingAddress,
		},
		
		// Now, successful v4 and v6 prefixes
		{
			"v4 prefix should be successful",
			&IPPrefix{
				Address: []byte{127,0,0,0},
				PrefixLen: 8,
				AddressFamily: IPVersion_IPv4,
			},
			netonly(net.ParseCIDR("127.0.0.1/8")),
			nil,
		},
		{
			"v6 prefix should be successful",
			&IPPrefix{
				Address: []byte{0xfe, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0xaf, 0xce, 0xa0, 0x0f, 0x42, 0xf5, 0x03},
				PrefixLen: 128,
				AddressFamily: IPVersion_IPv6,
			},
			netonly(net.ParseCIDR("fe80::af:cea0:f42:f503/128")),
			nil,
		},

		// Check that a gibberish address family returns an error
		{
			"bad AF should return error",
			&IPPrefix{
				Address: []byte{127,0,0,0},
				PrefixLen: 8,
				AddressFamily: 254,
			},
			net.IPNet{},
			ErrInvalidAddressFamily,
		},
		
		// Check for malformed addresses being caught
		{
			"malformed v6 addr should error",
			&IPPrefix{
				Address: []byte{0xfe, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
					// (address is several bytes short of a picnic)
				PrefixLen: 128,
				AddressFamily: IPVersion_IPv6,
			},
			net.IPNet{},
			ErrBadAddressForFamily,
		},
		{
			"malformed v4 addr should error",
			&IPPrefix{
				Address: []byte{0xfe, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
					// (address is several bytes short of a picnic)
				PrefixLen: 128,
				AddressFamily: IPVersion_IPv4,
			},
			net.IPNet{},
			ErrBadAddressForFamily,
		},


	}
	
	// Now we've got everything defined, we can run the tests.
	for _, test := range tests {
		runTest(test)
	}
	
}

func TestPrefixFromNetIPNet(t *testing.T) {
	// First of all, check roundtripability in v4 and v6
	// We do this by generating a load of random IP subnets and checking
	// that by passing them through PrefixFromIPNet and PrefixToIPNet returns
	// the same subnet that we first thought of.
	
	randomIPNet := func() net.IPNet {
		// random v4 or v6
		v4 := (rand.Intn(2) == 0)
		
		var addr net.IP
		var mask net.IPMask
		var addrLen int // address length to generate, in bytes
		var longestMask int // longest reasonable mask, in *bits*
		
		if (v4) {
			addrLen = 4
			longestMask = 24
		} else {
			addrLen = 16
			longestMask = 64
		}
		
		addr = make([]byte, addrLen, addrLen)
		for i := 0; i < addrLen; i++ {
			addr[i] = uint8(rand.Intn(256))
		}
		
		mask = net.CIDRMask(rand.Intn(longestMask + 1), addrLen * 8)
		
		return net.IPNet{
			IP: addr,
			Mask: mask,
		}
	}
	
	numberOfTrials := 50
	for i := 0; i < numberOfTrials; i++ {
		n := randomIPNet()
		var family IPVersion
		if len(n.IP) == 4 {
			family = IPVersion_IPv4
		} else {
			family = IPVersion_IPv6
		}
		
		pfx := PrefixFromNetIPNet(family, n)
		n2, _ := pfx.ToNetIPNet()
		
		if !reflect.DeepEqual(n, n2) {
			t.Errorf("Roundtripping failed; %v turned into %v", n, n2)
		}
	}
	
	// Now check some specific cases 
	
}
