package converters

import (
	serviceapi "code.justin.tv/amzn/TwitchExtensionsXenonTwirp"
	emsdocs "code.justin.tv/gds/gds/extensions/ems/documents"
)

func EMSInstalledExtensionsToXenon(input *emsdocs.InstalledExtensionsDocument) *serviceapi.InstalledExtensionsDocument {
	if input == nil {
		return nil
	}
	installedExtensions := make([]*serviceapi.InstalledExtension, len(input.InstalledExtensions))
	for index, installation := range input.InstalledExtensions {
		installedExtensions[index] = EMSInstalledExtensionToXenon(installation)
	}
	return &serviceapi.InstalledExtensionsDocument{
		InstalledExtensions: installedExtensions,
	}
}

func EMSInstalledExtensionToXenon(input *emsdocs.InstalledExtensionDocument) *serviceapi.InstalledExtension {
	if input == nil {
		return nil
	}

	output := &serviceapi.InstalledExtension{
		Extension:          EMSExtensionToXenonExtension(input.Extension),
		InstallationStatus: EMSInstallationStatusToXenon(input.InstallationStatus),
		RequiredActions:    make([]string, len(input.RequiredActions)),
	}
	for i, action := range input.RequiredActions {
		output.RequiredActions[i] = string(action)
	}
	return output
}

func EMSInstallationStatusToXenon(input *emsdocs.InstallationStatusDocument) *serviceapi.InstallationStatus {
	if input == nil {
		return nil
	}

	output := &serviceapi.InstallationStatus{
		ActivationState: input.ActivationState,
		CanActivate:     input.CanActivate,
		ID:              input.ID,
	}
	if input.Abilities != nil {
		output.Abilities = &serviceapi.InstallationAbilities{
			IsBitsEnabled:                 input.Abilities.IsBitsEnabled,
			IsChatEnabled:                 input.Abilities.IsChatEnabled,
			IsSubscriptionStatusAvailable: input.Abilities.IsSubscriptionStatusAvailable,
		}
	}
	if input.PermittedFeatures != nil {
		output.PermittedFeatures = &serviceapi.FeatureFlagsDocument{
			CanSendChat:                   input.PermittedFeatures.CanSendChat,
			CanUseBits:                    input.PermittedFeatures.CanUseBits,
			CanRetrieveSubscriptionStatus: input.PermittedFeatures.CanRetrieveSubscriptionStatus,
			DynamicManagement:             make([]*serviceapi.DynamicManagementParam, len(input.PermittedFeatures.DynamicManagement)),
		}

		for i, dynamicManagementParam := range input.PermittedFeatures.DynamicManagement {
			output.PermittedFeatures.DynamicManagement[i] = &serviceapi.DynamicManagementParam{
				GameID:    int32(dynamicManagementParam.GameID),
				IsManaged: dynamicManagementParam.IsManaged,
			}
		}
	}
	if input.Anchor != nil {
		if input.Anchor.Value == nil {
			output.Anchor = &serviceapi.InstallationStatus_Null{
				Null: &serviceapi.NullAnchorActivationParam{},
			}
		} else {
			switch input.Anchor.Value.Type() {
			case emsdocs.AnchorType("hidden"):
				output.Anchor = &serviceapi.InstallationStatus_Hidden{
					Hidden: &serviceapi.HiddenAnchorActivationParam{
						Slot: input.Anchor.Value.(*emsdocs.HiddenAnchorActivationParam).SlotName,
					},
				}
			case emsdocs.AnchorType("panel"):
				output.Anchor = &serviceapi.InstallationStatus_Panel{
					Panel: &serviceapi.PanelAnchorActivationParam{
						Slot: input.Anchor.Value.(*emsdocs.PanelAnchorActivationParam).SlotName,
					},
				}
			case emsdocs.AnchorType("video_overlay"):
				output.Anchor = &serviceapi.InstallationStatus_Overlay{
					Overlay: &serviceapi.VideoOverlayAnchorActivationParam{
						Slot: input.Anchor.Value.(*emsdocs.VideoOverlayAnchorActivationParam).SlotName,
					},
				}
			case emsdocs.AnchorType("component"):
				output.Anchor = &serviceapi.InstallationStatus_Component{
					Component: &serviceapi.ComponentAnchorActivationParam{
						Slot: input.Anchor.Value.(*emsdocs.ComponentAnchorActivationParam).SlotName,
						X:    int32(input.Anchor.Value.(*emsdocs.ComponentAnchorActivationParam).X),
						Y:    int32(input.Anchor.Value.(*emsdocs.ComponentAnchorActivationParam).Y),
					},
				}
			}
		}
	}
	return output
}

func EMSExtensionToXenonExtension(input *emsdocs.ExtensionDocument) *serviceapi.Extension {
	if input == nil {
		return nil
	}

	output := serviceapi.Extension{
		Anchor:                       input.Anchor,
		AssetURLs:                    make([]string, len(input.AssetURLs)),
		AuthorName:                   input.AuthorName,
		BaseURI:                      input.BaseURI,
		BitsEnabled:                  input.BitsEnabled,
		BitsSupportLevel:             string(input.BitsSupportLevel),
		CanInstall:                   input.CanInstall,
		Categories:                   make([]string, len(input.Categories)),
		ConfigurationLocation:        input.ConfigurationLocation,
		Description:                  input.Description,
		EULATOSURL:                   input.EULATOSURL,
		Games:                        make([]int32, len(input.Games)),
		ContentMatchedGames:          make([]int32, len(input.ContentMatchedGames)),
		HasChatSupport:               input.HasChatSupport,
		IconURLs:                     make(map[string]string),
		ID:                           input.ID,
		Name:                         input.Name,
		PrivacyPolicyURL:             input.PrivacyPolicyURL,
		RequestIdentityLink:          input.RequestIdentityLink,
		RequiredBroadcasterAbilities: make([]string, len(input.RequiredBroadcasterAbilities)),
		RequiredConfiguration:        input.RequiredConfiguration,
		ScreenshotURLs:               make([]string, len(input.ScreenshotURLs)),
		SKU:                          input.SKU,
		State:                        input.State,
		SubscriptionsSupportLevel:    string(input.SubscriptionsSupportLevel),
		Summary:                      input.Summary,
		SupportEmail:                 input.SupportEmail,
		VendorCode:                   input.VendorCode,
		Version:                      input.Version,
		ViewerSummary:                input.ViewerSummary,
		Views:                        &serviceapi.Extension_ExtensionViews{},
		WhitelistedConfigURLs:        make([]string, len(input.WhitelistedConfigURLs)),
		WhitelistedPanelURLs:         make([]string, len(input.WhitelistedPanelURLs)),
	}

	for i, assetURL := range input.AssetURLs {
		output.AssetURLs[i] = assetURL
	}

	for i, categories := range input.Categories {
		output.Categories[i] = categories
	}

	for i, game := range input.Games {
		output.Games[i] = int32(game)
	}

	for i, contentMatchedGame := range input.ContentMatchedGames {
		output.ContentMatchedGames[i] = int32(contentMatchedGame)
	}

	for key, value := range input.IconURLs {
		output.IconURLs[key] = value
	}

	for i, requiredBroadcasterAbility := range input.RequiredBroadcasterAbilities {
		output.RequiredBroadcasterAbilities[i] = requiredBroadcasterAbility
	}

	for i, screenshotURL := range input.ScreenshotURLs {
		output.ScreenshotURLs[i] = screenshotURL
	}

	if input.Views.Mobile != nil {
		output.Views.Mobile = &serviceapi.Extension_MobileExtensionView{
			CanLinkExternalContent: input.Views.Mobile.CanLinkExternalContent,
			ViewerPath:             input.Views.Mobile.ViewerPath,
			ViewerURL:              input.Views.Mobile.ViewerURL,
		}
	}

	if input.Views.Panel != nil {
		output.Views.Panel = &serviceapi.Extension_PanelExtensionView{
			CanLinkExternalContent: input.Views.Panel.CanLinkExternalContent,
			Height:                 int32(input.Views.Panel.Height),
			ViewerPath:             input.Views.Panel.ViewerPath,
			ViewerURL:              input.Views.Panel.ViewerURL,
		}
	}

	if input.Views.VideoOverlay != nil {
		output.Views.VideoOverlay = &serviceapi.Extension_VideoOverlayExtensionView{
			CanLinkExternalContent: input.Views.VideoOverlay.CanLinkExternalContent,
			ViewerPath:             input.Views.VideoOverlay.ViewerPath,
			ViewerURL:              input.Views.VideoOverlay.ViewerURL,
		}
	}

	if input.Views.Component != nil {
		output.Views.Component = &serviceapi.Extension_ComponentExtensionView{
			CanLinkExternalContent: input.Views.Component.CanLinkExternalContent,
			ViewerPath:             input.Views.Component.ViewerPath,
			ViewerURL:              input.Views.Component.ViewerURL,
			AspectRatioX:           int32(input.Views.Component.AspectRatioX),
			AspectRatioY:           int32(input.Views.Component.AspectRatioY),
			ScalePixels:            int32(input.Views.Component.ScalePixels),
			TargetHeight:           int32(input.Views.Component.TargetHeight),
			Autoscale:              input.Views.Component.Autoscale,
		}
	}

	if input.Views.Hidden != nil {
		output.Views.Hidden = &serviceapi.Extension_HiddenExtensionView{
			CanLinkExternalContent: input.Views.Hidden.CanLinkExternalContent,
			ViewerPath:             input.Views.Hidden.ViewerPath,
			ViewerURL:              input.Views.Hidden.ViewerURL,
		}
	}

	if input.Views.Config != nil {
		output.Views.Config = &serviceapi.Extension_ConfigExtensionView{
			CanLinkExternalContent: input.Views.Config.CanLinkExternalContent,
			ViewerPath:             input.Views.Config.ViewerPath,
			ViewerURL:              input.Views.Config.ViewerURL,
		}
	}

	if input.Views.LiveConfig != nil {
		output.Views.LiveConfig = &serviceapi.Extension_LiveConfigExtensionView{
			CanLinkExternalContent: input.Views.LiveConfig.CanLinkExternalContent,
			ViewerPath:             input.Views.LiveConfig.ViewerPath,
			ViewerURL:              input.Views.LiveConfig.ViewerURL,
		}
	}

	for i, whitelistedConfigURL := range input.WhitelistedConfigURLs {
		output.WhitelistedConfigURLs[i] = whitelistedConfigURL
	}

	for i, whitelistedPanelURL := range input.WhitelistedPanelURLs {
		output.WhitelistedPanelURLs[i] = whitelistedPanelURL
	}

	return &output
}
